package de.mklinger.qetch.client.model.v1.jackson;

import java.io.IOException;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;

import de.mklinger.qetch.client.common.MediaType;
import de.mklinger.qetch.client.common.SizeValue;
import de.mklinger.qetch.client.model.v1.Availability;
import de.mklinger.qetch.client.model.v1.AvailableConversion;
import de.mklinger.qetch.client.model.v1.Conversion;
import de.mklinger.qetch.client.model.v1.ConversionFile;
import de.mklinger.qetch.client.model.v1.Error;
import de.mklinger.qetch.client.model.v1.Job;
import de.mklinger.qetch.client.model.v1.NodeAddress;
import de.mklinger.qetch.client.model.v1.impl.AvailabilityImpl;
import de.mklinger.qetch.client.model.v1.impl.AvailableConversionImpl;
import de.mklinger.qetch.client.model.v1.impl.ConversionFileImpl;
import de.mklinger.qetch.client.model.v1.impl.ConversionImpl;
import de.mklinger.qetch.client.model.v1.impl.ErrorImpl;
import de.mklinger.qetch.client.model.v1.impl.JobImpl;
import de.mklinger.qetch.client.model.v1.impl.NodeAddressImpl;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class ObjectMapperConfigurer {
	public static ObjectMapper configure(final ObjectMapper objectMapper) {
		objectMapper.setSerializationInclusion(JsonInclude.Include.NON_ABSENT);
		objectMapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
		objectMapper.registerModule(new JavaTimeModule());
		objectMapper.registerModule(new QetchJacksonModule());
		return objectMapper;
	}

	private static class QetchJacksonModule extends SimpleModule {
		private static final long serialVersionUID = 1L;

		public QetchJacksonModule() {
			addSerializer(new JacksonMediaTypeSerializer());
			addDeserializer(MediaType.class, new JacksonMediaTypeDeserializer());
			addSerializer(new JacksonSizeValueSerializer());
			addDeserializer(SizeValue.class, new JacksonSizeValueDeserializer());

			addAbstractTypeMapping(NodeAddress.class, NodeAddressImpl.class);
			addAbstractTypeMapping(AvailableConversion.class, AvailableConversionImpl.class);
			addAbstractTypeMapping(Availability.class, AvailabilityImpl.class);
			addAbstractTypeMapping(Conversion.class, ConversionImpl.class);
			addAbstractTypeMapping(Job.class, JobImpl.class);
			addAbstractTypeMapping(ConversionFile.class, ConversionFileImpl.class);
			addAbstractTypeMapping(Error.class, ErrorImpl.class);
		}

		@Override
		public void setupModule(final SetupContext context) {
			context.insertAnnotationIntrospector(new QetchJacksonAnnotationIntrospector());
			super.setupModule(context);
		}
	}


	private static class JacksonMediaTypeSerializer extends StdSerializer<MediaType> {
		private static final long serialVersionUID = 1L;

		public JacksonMediaTypeSerializer() {
			super(MediaType.class);
		}

		@Override
		public void serialize(final MediaType value, final JsonGenerator gen, final SerializerProvider provider) throws IOException {
			gen.writeString(value.toString());
		}
	}

	private static class JacksonMediaTypeDeserializer extends StdDeserializer<MediaType> {
		private static final long serialVersionUID = 1L;

		public JacksonMediaTypeDeserializer() {
			super(MediaType.class);
		}

		@Override
		public MediaType deserialize(final JsonParser p, final DeserializationContext ctxt) throws IOException {
			return MediaType.valueOf(p.getText());
		}
	}

	private static class JacksonSizeValueSerializer extends StdSerializer<SizeValue> {
		private static final long serialVersionUID = 1L;

		public JacksonSizeValueSerializer() {
			super(SizeValue.class);
		}

		@Override
		public void serialize(final SizeValue value, final JsonGenerator gen, final SerializerProvider provider) throws IOException {
			gen.writeNumber(value.getSingles());
		}
	}

	private static class JacksonSizeValueDeserializer extends StdDeserializer<SizeValue> {
		private static final long serialVersionUID = 1L;

		public JacksonSizeValueDeserializer() {
			super(SizeValue.class);
		}

		@Override
		public SizeValue deserialize(final JsonParser p, final DeserializationContext ctxt) throws IOException {
			return new SizeValue(p.getValueAsLong());
		}
	}
}
