package de.mklinger.qetcher.client.model.v1.impl;

import java.math.BigInteger;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Objects;

import de.mklinger.micro.lists.Lists;
import de.mklinger.qetcher.client.model.v1.ConversionFile;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.client.model.v1.ModelValidationException;
import de.mklinger.qetcher.client.model.v1.builder.ConversionFileBuilder;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class ConversionFileImpl implements ConversionFile {
	private final String fileId;
	private final boolean serviceManaged;
	private final String tenantId;
	private final String name;
	private final BigInteger size;
	private final MediaType mediaType;
	private final List<String> nodeIds;
	private final Duration deleteTimeout;
	private final Instant created;
	private final Instant updated;

	public ConversionFileImpl(final ConversionFileBuilder builder) {
		this.fileId = builder.getFileId();
		this.serviceManaged = builder.isServiceManaged();
		this.tenantId = builder.getTenantId();
		this.name = builder.getName();
		this.size = builder.getSize();
		this.mediaType = builder.getMediaType();
		this.nodeIds = Lists.newImmutableList(builder.getNodeIds());
		this.deleteTimeout = builder.getDeleteTimeout();
		this.created = builder.getCreated();
		this.updated = builder.getUpdated();

		try {
			validate();
		} catch (final Exception e) {
			throw new ModelValidationException(ConversionFile.class, e);
		}
	}

	private void validate() {
		Objects.requireNonNull(fileId);
		Objects.requireNonNull(tenantId);
		Objects.requireNonNull(size);
		Objects.requireNonNull(mediaType);
		Objects.requireNonNull(nodeIds);
		if (nodeIds.isEmpty()) {
			throw new IllegalStateException("nodeIds must not be empty");
		}
		Objects.requireNonNull(created);
		Objects.requireNonNull(updated);
	}

	@Override
	public String getFileId() {
		return fileId;
	}

	@Override
	public boolean isServiceManaged() {
		return serviceManaged;
	}

	@Override
	public String getTenantId() {
		return tenantId;
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public BigInteger getSize() {
		return size;
	}

	@Override
	public MediaType getMediaType() {
		return mediaType;
	}

	@Override
	public List<String> getNodeIds() {
		return nodeIds;
	}

	@Override
	public Duration getDeleteTimeout() {
		return deleteTimeout;
	}

	@Override
	public Instant getCreated() {
		return created;
	}

	@Override
	public Instant getUpdated() {
		return updated;
	}

	@Override
	public String toString() {
		return "ConversionFileImpl{" +
				"fileId='" + fileId + '\'' +
				", serviceManaged=" + serviceManaged +
				", tenantId='" + tenantId + '\'' +
				", name='" + name + '\'' +
				", size=" + size +
				", mediaType=" + mediaType +
				", nodeIds=" + nodeIds +
				", deleteTimeout=" + deleteTimeout +
				", created=" + created +
				", updated=" + updated +
				'}';
	}
}
