package de.mklinger.qetcher.liferay.client.impl.liferay71;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.document.library.kernel.document.conversion.DocumentConversionUtil;
import com.liferay.document.library.kernel.exception.NoSuchFileEntryException;
import com.liferay.document.library.kernel.util.DLUtil;
import com.liferay.document.library.kernel.util.PDFProcessor;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.repository.model.FileVersion;
import com.liferay.portal.kernel.util.FileUtil;

import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.client.model.v1.MediaTypes;
import de.mklinger.qetcher.liferay.client.QetcherService;

@Component(
		immediate = true,
		property = {
				"qetcher:Boolean=true",
				"service.ranking:Integer=1000"
		})
public class QetcherPDFProcessorImpl extends QetcherPDFProcessorImplBoilerplate implements PDFProcessor {
	private static final Logger LOG = LoggerFactory.getLogger(QetcherPDFProcessorImpl.class);

	private QetcherService qetcherService;

	@Reference
	public void setQetcherService(QetcherService qetcherService) {
		this.qetcherService = qetcherService;
	}

	@Activate
	@Override
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
	}

	@Override
	public void generateImages(FileVersion sourceFileVersion, FileVersion destinationFileVersion) throws Exception {
		try {

			if (sourceFileVersion != null) { // Looks strange. Taken from original implementation.
				copy(sourceFileVersion, destinationFileVersion);

				return;
			}

			if (_hasImages(destinationFileVersion)) {
				return;
			}

			final String extension = destinationFileVersion.getExtension();

			if (extension.equals("pdf")) {
				try (InputStream inputStream = destinationFileVersion.getContentStream(false)) {
					generateImages(destinationFileVersion, inputStream);
				}
			} else if (DocumentConversionUtil.isEnabled()) {
				// More or less original impl code follows:
				final String tempFileId = DLUtil.getTempFileId(
						destinationFileVersion.getFileEntryId(),
						destinationFileVersion.getVersion());

				if (Objects.equals(
						"PWC", destinationFileVersion.getVersion()) ||
						destinationFileVersion.isPending()) {

					final File file = new File(
							DocumentConversionUtil.getFilePath(
									tempFileId, "pdf"));

					FileUtil.delete(file);
				}

				File pdfFile;

				try (InputStream inputStream = destinationFileVersion.getContentStream(false)) {
					pdfFile = DocumentConversionUtil.convert(
							tempFileId, inputStream, extension, "pdf");
				} catch (final Exception e) {
					LOG.info("Conversion from {} to {} failed. Not generating images.", extension, "pdf");
					pdfFile = null;
				}

				if (pdfFile != null) {
					try (FileInputStream in = new FileInputStream(pdfFile)) {
						generateImages(destinationFileVersion, in);
					}
				}
			}

		} catch (final NoSuchFileEntryException nsfee) {
			if (LOG.isDebugEnabled()) {
				LOG.debug("File not found", nsfee);
			}
		} finally {
			fileVersionIds.remove(destinationFileVersion.getFileVersionId());
		}
	}

	private void generateImages(FileVersion fileVersion, InputStream pdfInputStream) throws Exception {
		final MediaType fromMediaType = MediaTypes.PDF;
		final MediaType toMediaType = MediaTypes.PNG
				.withParameter("resolution", String.valueOf(getPreviewDocumentDocumentDpi()));

		final String tempFileId = getTempFileId(fileVersion);
		final String referrer = "liferay-pdfprocessor-id=" + tempFileId;

		if (_isGeneratePreview(fileVersion)) {
			// As in PDFProcessorImpl._generateImagesGS(FileVersion, File, boolean):
			final Function<Integer, Path> targetFileSupplier = idx -> getPreviewTempFile(tempFileId, idx + 2).toPath();

			final List<Path> previewFiles = qetcherService.convertToFiles(pdfInputStream, targetFileSupplier, fromMediaType, toMediaType, referrer);

			// In contrast to Liferay implementation, we re-use the first page for thumbnail
			if (_isGenerateThumbnail(fileVersion) && !previewFiles.isEmpty()) {
				storeThumbnailImages(fileVersion, previewFiles.get(0).toFile());
			}

			addPreviewFilesToStore(fileVersion, tempFileId);

		} else if (_isGenerateThumbnail(fileVersion)) {
			final MediaType thumbnailToMediaType = toMediaType
					.withParameter("pages", "1");

			final Path targetFile = getThumbnailTempFile(tempFileId).toPath();

			qetcherService.convertToFile(pdfInputStream, targetFile, fromMediaType, thumbnailToMediaType, referrer);

			addThumbnailToStore(fileVersion, tempFileId);
		}
	}

	private void addThumbnailToStore(FileVersion fileVersion, String tempFileId) throws Exception {
		final File thumbnailTempFile = getThumbnailTempFile(tempFileId);

		try {
			storeThumbnailImages(fileVersion, thumbnailTempFile);
		}
		finally {
			FileUtil.delete(thumbnailTempFile);
		}
	}

	private String getTempFileId(FileVersion fileVersion) {
		return DLUtil.getTempFileId(fileVersion.getFileEntryId(), fileVersion.getVersion());
	}

	// As in PDFProcessorImpl._generateImagesGS(FileVersion, File, boolean)
	private void addPreviewFilesToStore(FileVersion fileVersion, final String tempFileId) throws PortalException {
		final int total = getPreviewTempFileCount(fileVersion);

		for (int i = 0; i < total; i++) {
			final File previewTempFile = getPreviewTempFile(tempFileId, i + 2);

			try {
				final String previewFilePath = getPreviewFilePath(fileVersion, i + 1);

				LOG.debug("Adding preview file to store: idx {}, file {}", i, previewFilePath);

				addFileToStore(
						fileVersion.getCompanyId(), PREVIEW_PATH,
						previewFilePath,
						previewTempFile);
			}
			finally {
				FileUtil.delete(previewTempFile);
			}
		}
	}
}
