package de.mklinger.qetcher.liferay.client.impl.liferay71;

import java.io.File;
import java.io.InputStream;
import java.util.Map;
import java.util.Set;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.document.library.kernel.exception.NoSuchFileEntryException;
import com.liferay.document.library.kernel.util.AudioProcessor;
import com.liferay.document.library.kernel.util.DLUtil;
import com.liferay.portal.configuration.metatype.bnd.util.ConfigurableUtil;
import com.liferay.portal.kernel.repository.model.FileVersion;
import com.liferay.portal.kernel.util.FileUtil;

import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.liferay.client.QetcherConversionsService;
import de.mklinger.qetcher.liferay.client.QetcherService;
import de.mklinger.qetcher.liferay.client.impl.QetcherConfiguration;

@Component(
		immediate = true,
		property = {
				"qetcher:Boolean=true",
				"service.ranking:Integer=1000"
		},
		configurationPid = QetcherConfiguration.ID,
		configurationPolicy = ConfigurationPolicy.REQUIRE)
public class QetcherAudioProcessorImpl extends QetcherAudioProcessorImplBoilerplate implements AudioProcessor {
	private static final Logger LOG = LoggerFactory.getLogger(QetcherAudioProcessorImpl.class);

	private QetcherService qetcherService;
	private QetcherConversionsService qetcherConversionsService;
	private ProcessorMediaTypes processorMediaTypes;
	private QetcherConfiguration qetcherConfiguration;

	@Reference
	public void setQetcherService(QetcherService qetcherService) {
		this.qetcherService = qetcherService;
	}

	@Reference
	public void setQetcherConversionsService(QetcherConversionsService qetcherConversionsService) {
		this.qetcherConversionsService = qetcherConversionsService;
	}

	@Reference
	public void setProcessorMediaTypes(ProcessorMediaTypes processorMediaTypes) {
		this.processorMediaTypes = processorMediaTypes;
	}

	@Activate
	@Modified
	public synchronized void activate(final Map<String, Object> configurationProperties) throws Exception {
		LOG.debug("Configuration was set");
		setQetcherConfiguration(ConfigurableUtil.createConfigurable(QetcherConfiguration.class, configurationProperties));
		super.afterPropertiesSet();
	}

	@Reference
	public void setQetcherConfiguration(QetcherConfiguration qetcherConfiguration) {
		this.qetcherConfiguration = qetcherConfiguration;
	}

	@Override
	public boolean isSupported(String mimeType) {
		if (getAudioMimeTypes().contains(mimeType)) {
			LOG.debug("Supported audio type: {}", mimeType);
			return true;
		}

		LOG.debug("Unsupported audio type: {}", mimeType);
		return false;
	}

	@Override
	public Set<String> getAudioMimeTypes() {
		if (qetcherConfiguration.useAllSupportedAudioTypes()) {
			return qetcherConversionsService.getSourceMediaTypesForRegistry("audio");
		} else {
			return liferayAudioMimeTypes;
		}
	}

	@Override
	public void generateAudio(FileVersion sourceFileVersion, FileVersion destinationFileVersion) throws Exception {
		try {
			if (sourceFileVersion != null) {
				copy(sourceFileVersion, destinationFileVersion);

				return;
			}

			if (_hasAudio(destinationFileVersion)) {
				return;
			}

			if (!hasPreviews(destinationFileVersion)) {
				for (final String previewType : _PREVIEW_TYPES) {
					try (InputStream inputStream = destinationFileVersion.getContentStream(false)) {

						generateAudioQetcher(destinationFileVersion, inputStream, previewType);

					} catch (final NoSuchFileEntryException nsfee) {
						if (LOG.isDebugEnabled()) {
							LOG.debug("File entry not found", nsfee);
						}
					} catch (final Exception e) {
						LOG.error("Error generating audio for file version " + destinationFileVersion.getFileVersionId(), e);
					}
				}
			}
		} catch (final NoSuchFileEntryException nsfee) {
			if (LOG.isDebugEnabled()) {
				LOG.debug("File entry not found", nsfee);
			}
		} finally {
			_fileVersionIds.remove(destinationFileVersion.getFileVersionId());
		}
	}

	private void generateAudioQetcher(FileVersion fileVersion, InputStream inputStream, String containerType) throws Exception {
		if (hasPreview(fileVersion, containerType)) {
			return;
		}

		final String tempFileId = DLUtil.getTempFileId(
				fileVersion.getFileEntryId(),
				fileVersion.getVersion());

		final File previewTempFile = getPreviewTempFile(tempFileId, containerType);

		try {

			final MediaType fromMediaType = processorMediaTypes.getMediaType(fileVersion);
			final MediaType toMediaType = processorMediaTypes.getContainerMediaType(containerType);
			final String referrer = "liferay-audioprocessor-id=" + tempFileId;

			LOG.info("Converting file version {} from {} to {}", fileVersion.getFileVersionId(), fromMediaType, toMediaType);

			qetcherService.convertToFile(inputStream, previewTempFile.toPath(), fromMediaType, toMediaType, referrer);

			addFileToStore(
					fileVersion.getCompanyId(), PREVIEW_PATH,
					getPreviewFilePath(fileVersion, containerType), previewTempFile);

		} finally {
			FileUtil.delete(previewTempFile);
		}
	}
}
