package de.mklinger.qetcher.liferay.client.impl;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.TimeUnit;

import org.osgi.service.component.annotations.Component;

@Component(service = QetcherTimeoutConfig.class)
public class QetcherTimeoutConfig {
	private final TimeValue shortTimeout = TimeValue.seconds(30);
	private final TimeValue uploadTimeout = TimeValue.hours(1);
	private final TimeValue downloadTimeout = TimeValue.hours(1);
	private final TimeValue jobCreateTimeout = TimeValue.seconds(30);
	private final TimeValue jobWaitTimeout = TimeValue.hours(1);
	private final TimeValue jobDeleteTimeout = TimeValue.seconds(30);
	private final TimeValue jobServiceCancelTimeout = jobWaitTimeout.plus(TimeValue.minutes(1));
	private final TimeValue jobServiceDeleteTimeout = jobServiceCancelTimeout.plus(TimeValue.minutes(30));
	private final TimeValue fileServiceDeleteTimeout = jobServiceDeleteTimeout;

	public TimeValue getShortTimeout() {
		return shortTimeout;
	}

	public TimeValue getUploadTimeout() {
		return uploadTimeout;
	}

	public TimeValue getDownloadTimeout() {
		return downloadTimeout;
	}

	public TimeValue getJobCreateTimeout() {
		return jobCreateTimeout;
	}

	public TimeValue getJobWaitTimeout() {
		return jobWaitTimeout;
	}

	public TimeValue getJobDeleteTimeout() {
		return jobDeleteTimeout;
	}

	public TimeValue getJobServiceCancelTimeout() {
		return jobServiceCancelTimeout;
	}

	public TimeValue getJobServiceDeleteTimeout() {
		return jobServiceDeleteTimeout;
	}

	public TimeValue getFileServiceDeleteTimeout() {
		return fileServiceDeleteTimeout;
	}

	public static class TimeValue {
		private final long time;
		private final TimeUnit timeUnit;

		public TimeValue(long time, TimeUnit timeUnit) {
			this.time = time;
			this.timeUnit = timeUnit;
		}

		public static TimeValue seconds(long seconds) {
			return new TimeValue(seconds, TimeUnit.SECONDS);
		}

		public static TimeValue minutes(long minutes) {
			return new TimeValue(minutes, TimeUnit.MINUTES);
		}

		public static TimeValue hours(long hours) {
			return new TimeValue(hours, TimeUnit.HOURS);
		}

		public TimeValue plus(TimeValue timeValue) {
			return of(toDuration().plus(timeValue.toDuration()));
		}

		public static TimeValue of(Duration duration) {
			if (duration.getSeconds() > 0) {
				try {
					return new TimeValue(duration.toMillis(), TimeUnit.MILLISECONDS);
				} catch (final ArithmeticException e) {
					return new TimeValue(duration.getSeconds(), TimeUnit.SECONDS);
				}
			} else {
				return new TimeValue(duration.getNano(), TimeUnit.NANOSECONDS);
			}
		}

		public long time() {
			return time;
		}

		public TimeUnit unit() {
			return timeUnit;
		}

		public Duration toDuration() {
			switch (timeUnit) {
			case NANOSECONDS:
				return Duration.of(time, ChronoUnit.NANOS);
			case MICROSECONDS:
				return Duration.of(time, ChronoUnit.MICROS);
			case SECONDS:
				return Duration.of(time, ChronoUnit.SECONDS);
			case MINUTES:
				return Duration.of(time, ChronoUnit.MINUTES);
			case HOURS:
				return Duration.of(time, ChronoUnit.HOURS);
			case DAYS:
				return Duration.of(time, ChronoUnit.DAYS);
			default:
				return Duration.of(timeUnit.toMillis(time), ChronoUnit.MILLIS);
			}

		}
	}
}
