package de.mklinger.qetcher.liferay.client.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.function.Function;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.mklinger.qetcher.client.InputConversionFile;
import de.mklinger.qetcher.client.InputJob;
import de.mklinger.qetcher.client.model.v1.AvailableConversion;
import de.mklinger.qetcher.client.model.v1.Job;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.client.model.v1.MediaTypes;
import de.mklinger.qetcher.htmlinliner.HtmlElementInliner;
import de.mklinger.qetcher.liferay.client.QetcherClientService;
import de.mklinger.qetcher.liferay.client.QetcherService;
import de.mklinger.qetcher.liferay.client.impl.abstraction.liferay71.LiferayAbstractionFactorySupplier;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
@Component
public class QetcherServiceImpl implements QetcherService {
	private static final Logger LOG = LoggerFactory.getLogger(QetcherServiceImpl.class);

	private QetcherClientService clientService;
	private QetcherJobs qetcherJobs;
	private QetcherTimeoutConfig timeoutConfig;

	@Reference
	public void setClientService(QetcherClientService clientService) {
		this.clientService = clientService;
	}

	@Reference
	public void setQetcherJobs(QetcherJobs qetcherJobs) {
		this.qetcherJobs = qetcherJobs;
	}

	@Reference
	public void setTimeoutConfig(QetcherTimeoutConfig timeoutConfig) {
		this.timeoutConfig = timeoutConfig;
	}

	@Override
	public Path convertToFile(InputStream inputStream, Path targetFile, MediaType fromMediaType, MediaType toMediaType, String referrer) {
		try {

			return doConvert(
					inputStream,
					fromMediaType,
					toMediaType,
					referrer,
					fileId -> clientService.client().downloadAsFile(fileId, targetFile));

		} catch(final CompletionException e) {
			throw unwrap(e);
		}
	}

	@Override
	public List<Path> convertToFiles(InputStream inputStream, Function<Integer, Path> targetFileSupplier, MediaType fromMediaType, MediaType toMediaType, String referrer) {
		try {

			return doConvert(
					inputStream,
					fromMediaType,
					toMediaType,
					referrer,
					Job::getResultFileIds,
					resultFileIds -> this.downloadAll(resultFileIds, targetFileSupplier));

		} catch(final CompletionException e) {
			throw unwrap(e);
		}
	}

	@Override
	public Path convertToTempFile(final InputStream inputStream, final MediaType fromMediaType, final MediaType toMediaType, String referrer) {
		try {

			return doConvert(
					inputStream,
					fromMediaType,
					toMediaType,
					referrer,
					clientService.client()::downloadAsTempFile);

		} catch(final CompletionException e) {
			throw unwrap(e);
		}
	}

	@Override
	public byte[] convertToByteArray(InputStream inputStream, MediaType fromMediaType, MediaType toMediaType, String referrer) {
		try {

			return doConvert(
					inputStream,
					fromMediaType,
					toMediaType,
					referrer,
					clientService.client()::downloadAsByteArray);

		} catch(final CompletionException e) {
			throw unwrap(e);
		}
	}

	private <T> T doConvert(final InputStream inputStream, final MediaType fromMediaType, final MediaType toMediaType, String referrer,
			Function<String, ? extends CompletionStage<T>> jobSuccessAction) {

		return doConvert(inputStream, fromMediaType, toMediaType, referrer, qetcherJobs::getSingleResultFileId, jobSuccessAction);

	}

	private <T, X> T doConvert(final InputStream inputStream, final MediaType fromMediaType, final MediaType toMediaType, String referrer,
			Function<Job, X> jobMapping,
			Function<X, ? extends CompletionStage<T>> jobSuccessAction) {

		final InputStream actualInputStream = getActualInputStream(inputStream, fromMediaType);

		final InputJob inputJob = newInputJob(actualInputStream, fromMediaType, toMediaType, referrer);

		final Job job = qetcherJobs.createJobWithTimeout(inputJob)
				.join();

		try {

			return qetcherJobs.getJobDoneWithTimeout(job)
					.thenApply(qetcherJobs::requireSuccess)
					.thenApply(jobMapping)
					.thenCompose(jobSuccessAction)
					.join();

		} finally {
			qetcherJobs.deleteJobWithTimeout(job)
			.exceptionally(e -> {
				LOG.warn("Error deleting job", e);
				return null;
			});
		}
	}

	private InputJob newInputJob(final InputStream actualInputStream, final MediaType from, final MediaType to, String referrer) {
		final InputConversionFile inputFile = clientService.inputFileFor(actualInputStream)
				.mediaType(from)
				.build();

		final InputJob inputJob = clientService.job()
				.fromFile(inputFile)
				.toMediaType(to)
				.referrer(referrer)
				.cancelTimeout(timeoutConfig.getJobServiceCancelTimeout().toDuration())
				.deleteTimeout(timeoutConfig.getJobServiceDeleteTimeout().toDuration())
				.build();

		return inputJob;
	}

	private InputStream getActualInputStream(final InputStream inputStream, final MediaType from) {
		if (MediaTypes.HTML.isCompatible(from)) {
			return inlineHtmlElements(inputStream);
		} else {
			return inputStream;
		}
	}

	private InputStream inlineHtmlElements(final InputStream inputStream) {
		LOG.info("Inlining HTML elements...");

		final String baseUri = LiferayAbstractionFactorySupplier.getInstance().getPortalTool().getBaseUrl();
		if (baseUri == null) {
			LOG.warn("Could not get base URL - no HTML inlining can be done");
			return inputStream;
		}

		byte[] data;

		try (InputStream in = inputStream) {
			try (HtmlElementInliner inliner = LiferayHtmlElementInlinerFactory.newHtmlElementInliner()) {
				data = inliner.inline(in, baseUri);
			}
		} catch (final IOException e) {
			throw new UncheckedIOException(e);
		}

		return new ByteArrayInputStream(data);
	}

	private CompletableFuture<List<Path>> downloadAll(List<String> resultFileIds, Function<Integer, Path> targetFileSupplier) {
		final List<Path> resultFiles = new ArrayList<>(resultFileIds.size());

		int idx = 0;
		for (final String fileId : resultFileIds) {
			final Path targetFile = targetFileSupplier.apply(idx);
			// join is ugly here, but ok for bulk download
			clientService.client().downloadAsFile(fileId, targetFile).join();
			idx++;
		}

		return CompletableFuture.completedFuture(resultFiles);
	}

	private static RuntimeException unwrap(CompletionException e) {
		final Throwable cause = e.getCause();
		if (cause != null && cause instanceof RuntimeException) {
			return (RuntimeException)cause;
		} else {
			return e;
		}
	}

	@Override
	public List<AvailableConversion> getAvailableConversions() {
		return clientService.client().getAvailableConversions().join();
	}
}
