package de.mklinger.qetcher.liferay.client.impl;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.mklinger.qetcher.client.InputJob;
import de.mklinger.qetcher.client.QetcherClientException;
import de.mklinger.qetcher.client.QetcherRemoteException;
import de.mklinger.qetcher.client.common.concurrent.Delay;
import de.mklinger.qetcher.client.model.v1.Job;
import de.mklinger.qetcher.client.model.v1.JobState;
import de.mklinger.qetcher.liferay.client.QetcherClientService;
import de.mklinger.qetcher.liferay.client.impl.QetcherTimeoutConfig.TimeValue;

@Component(service = QetcherJobs.class)
public class QetcherJobs {
	private static final Logger LOG = LoggerFactory.getLogger(QetcherJobs.class);

	private QetcherClientService clientService;
	private QetcherTimeoutConfig timeoutConfig;

	@Reference
	public void setClientService(QetcherClientService clientService) {
		this.clientService = clientService;
	}

	@Reference
	public void setTimeoutConfig(QetcherTimeoutConfig timeoutConfig) {
		this.timeoutConfig = timeoutConfig;
	}

	public CompletableFuture<Job> createJobWithTimeout(final InputJob inputJob) {
		final TimeValue timeout = timeoutConfig.getJobCreateTimeout();

		return Delay.timeout(
				clientService.client().createJob(inputJob),
				timeout.time(), timeout.unit(),
				() -> new TimeoutException(
						"Timeout creating job: "
								+ inputJob.getFromMediaType()
								+ " -> "
								+ inputJob.getToMediaType()));
	}

	public CompletableFuture<Job> getJobDoneWithTimeout(final Job job) {
		final TimeValue timeout = timeoutConfig.getJobWaitTimeout();

		return Delay.timeout(
				clientService.client().getJobDone(job),
				timeout.time(), timeout.unit(),
				() -> new TimeoutException(
						"Timeout getting job done "
								+ job.getJobId()
								+ " "
								+ job.getFromMediaType()
								+ " -> "
								+ job.getToMediaType()));
	}

	public Job requireSuccess(final Job job) {
		if (job.getState() != JobState.SUCCESS) {
			Exception cause;
			if (job.getError() != null) {
				cause = new QetcherRemoteException(job.getError(), -1);
			} else {
				cause = null;
			}
			throw new QetcherClientException("Job " + job.getJobId() + " is not in state SUCCESS but " + job.getState(), cause);
		}

		return job;
	}

	public CompletableFuture<Void> deleteJobWithTimeout(final Job job) {
		final TimeValue timeout = timeoutConfig.getJobDeleteTimeout();

		return Delay.timeout(
				clientService.client().deleteJob(job),
				timeout.time(), timeout.unit(),
				() -> new TimeoutException(
						"Timeout deleting job "
								+ job.getJobId()
								+ " "
								+ job.getFromMediaType()
								+ " -> "
								+ job.getToMediaType()));
	}

	public String getSingleResultFileId(final Job job) {
		final List<String> resultFileIds = job.getResultFileIds();
		if (resultFileIds == null || resultFileIds.isEmpty()) {
			throw new QetcherClientException("Expected 1 result file but have no result files");
		}
		if (resultFileIds.size() != 1) {
			LOG.warn("Expected 1 result file but is " + resultFileIds.size());
		}
		return resultFileIds.get(0);
	}
}
