package de.mklinger.qetcher.liferay.client.impl.liferay71.scr;

import java.util.HashSet;
import java.util.Set;
import java.util.function.Consumer;

import org.osgi.framework.BundleContext;
import org.osgi.framework.Filter;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTracker;

/**
 * Service tracker with callbacks for service availability and unavailability.
 *
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class ServiceAvailableTracker<S> implements AutoCloseable {
	private final ServiceTracker<S, S> tracker;

	public ServiceAvailableTracker(BundleContext bundleContext, Filter filter, Consumer<ServiceReference<S>> onServiceAvailable, Runnable onServiceUnavailable) {
		// Set for known service references. Two ServiceReference objects will have same
		// hashCode and will be equal for same service registration.
		// See ServiceReference class documentation.
		final Set<ServiceReference<S>> knownReferences = new HashSet<>();

		tracker = new ServiceTracker<S, S>(bundleContext, filter, null) {
			@Override
			public S addingService(ServiceReference<S> reference) {
				int countBefore;

				synchronized (knownReferences) {
					countBefore = knownReferences.size();
					knownReferences.add(reference);
				}

				if (countBefore == 0) {
					onServiceAvailable.accept(reference);
				}

				// Don't track the actual service instance
				return null;
			}

			@Override
			public void removedService(ServiceReference<S> reference, S service) {
				int countAfter;

				synchronized (knownReferences) {
					knownReferences.remove(reference);
					countAfter = knownReferences.size();
				}

				if (countAfter == 0) {
					onServiceUnavailable.run();
				}
			}
		};
	}

	/**
	 * Open this {@code ServiceTracker} and begin tracking services.
	 *
	 * @throws java.lang.IllegalStateException If the {@code BundleContext} with
	 *         which this {@code ServiceTracker} was created is no longer valid.
	 * @see #open(boolean)
	 */
	public void open() {
		tracker.open();
	}

	/**
	 * Open this {@code ServiceTracker} and begin tracking services.
	 *
	 * <p>
	 * Services which match the search criteria specified when this
	 * {@code ServiceTracker} was created are now tracked by this
	 * {@code ServiceTracker}.
	 *
	 * @param trackAllServices If {@code true}, then this {@code ServiceTracker}
	 *        will track all matching services regardless of class loader
	 *        accessibility. If {@code false}, then this {@code ServiceTracker}
	 *        will only track matching services which are class loader
	 *        accessible to the bundle whose {@code BundleContext} is used by
	 *        this {@code ServiceTracker}.
	 * @throws java.lang.IllegalStateException If the {@code BundleContext} with
	 *         which this {@code ServiceTracker} was created is no longer valid.
	 */
	public void open(boolean trackAllServices) {
		tracker.open(trackAllServices);
	}

	/**
	 * Close this {@code ServiceTracker}.
	 *
	 * <p>
	 * This method should be called when this {@code ServiceTracker} should end
	 * the tracking of services.
	 */
	@Override
	public void close() {
		tracker.close();
	}
}