package de.mklinger.qetcher.liferay.client.impl.liferay71;

import java.io.InputStream;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.document.library.kernel.exception.NoSuchFileEntryException;
import com.liferay.document.library.kernel.util.DLUtil;
import com.liferay.document.library.kernel.util.VideoProcessor;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.repository.model.FileVersion;
import com.liferay.portal.kernel.util.FileUtil;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.PropsKeys;
import com.liferay.portal.kernel.util.PropsUtil;

import de.mklinger.qetcher.client.InputConversionFile;
import de.mklinger.qetcher.client.InputJob;
import de.mklinger.qetcher.client.model.v1.ConversionFile;
import de.mklinger.qetcher.client.model.v1.Job;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.liferay.client.QetcherClientService;
import de.mklinger.qetcher.liferay.client.QetcherConversionsService;
import de.mklinger.qetcher.liferay.client.impl.QetcherJobs;

@Component(
		immediate = true,
		property = {
				"qetcher:Boolean=true",
				"service.ranking:Integer=1000"
		})
public class QetcherVideoProcessorImpl extends QetcherVideoProcessorImplBoilerplate implements VideoProcessor {
	private static final Logger LOG = LoggerFactory.getLogger(QetcherVideoProcessorImpl.class);

	private QetcherClientService qetcherClientService;
	private QetcherJobs qetcherJobs;
	private QetcherConversionsService qetcherConversionsService;
	private ProcessorMediaTypes processorMediaTypes;

	@Reference
	public void setQetcherClientService(QetcherClientService qetcherClientService) {
		this.qetcherClientService = qetcherClientService;
	}

	@Reference
	public void setQetcherJobs(QetcherJobs qetcherJobs) {
		this.qetcherJobs = qetcherJobs;
	}

	@Reference
	public void setQetcherConversionsService(QetcherConversionsService qetcherConversionsService) {
		this.qetcherConversionsService = qetcherConversionsService;
	}

	@Reference
	public void setProcessorMediaTypes(ProcessorMediaTypes processorMediaTypes) {
		this.processorMediaTypes = processorMediaTypes;
	}

	@Activate
	@Override
	public void afterPropertiesSet() {
		super.afterPropertiesSet();
	}

	@Override
	public Set<String> getVideoMimeTypes() {
		return qetcherConversionsService.getSourceMediaTypesForRegistry("video");
	}

	@Override
	public void generateVideo(FileVersion sourceFileVersion, FileVersion destinationFileVersion) throws Exception {
		if (_hasVideo(destinationFileVersion)) {
			return;
		}

		final AtomicReference<ConversionFile> conversionFile = new AtomicReference<>();
		final List<Job> jobs = new ArrayList<>();
		final List<Path> tempFiles = new ArrayList<>();

		try {
			if (sourceFileVersion != null) {
				copy(sourceFileVersion, destinationFileVersion);

				return;
			}

			if (!hasPreviews(destinationFileVersion) || !hasThumbnails(destinationFileVersion)) {

				final MediaType fromMediaType = processorMediaTypes.getMediaType(destinationFileVersion);

				LOG.info("Uploading file version {} with media type {}", destinationFileVersion.getFileVersionId(), fromMediaType);

				try (InputStream inputStream = getContentStream(destinationFileVersion)) {
					// TODO use timeouts from configuration
					final InputConversionFile inputConversionFile = qetcherClientService.inputFileFor(inputStream)
							.mediaType(fromMediaType)
							.build();

					conversionFile.set(qetcherClientService.client().uploadFile(inputConversionFile)
							.join());
				}

				final List<CompletableFuture<Void>> jobFutures = new ArrayList<>();

				if (!hasPreviews(destinationFileVersion)) {
					final String tempFileId = DLUtil.getTempFileId(
							destinationFileVersion.getFileEntryId(),
							destinationFileVersion.getVersion());

					for (final String previewType : _PREVIEW_TYPES) {

						final MediaType toMediaType = processorMediaTypes.getContainerMediaType(previewType);
						final String referrer = "liferay-videoprocessor-id=" + tempFileId + ";liferay-videoprocessor-type=" + previewType;

						final Job job = newJob(conversionFile.get(), toMediaType, referrer);
						jobs.add(job);

						final Path targetFile = getPreviewTempFile(tempFileId, previewType).toPath();
						tempFiles.add(targetFile);

						final CompletableFuture<Void> jobFuture = runJob(job, targetFile)
								.thenRunAsync(() -> addFileToStore(
										destinationFileVersion.getCompanyId(), PREVIEW_PATH,
										getPreviewFilePath(destinationFileVersion, previewType), targetFile));

						jobFutures.add(jobFuture);
					}
				}

				if (!hasThumbnails(destinationFileVersion)) {

					final String tempFileId = DLUtil.getTempFileId(destinationFileVersion.getFileEntryId(), destinationFileVersion.getVersion());

					final Path thumbnailTempFile = getThumbnailTempFile(tempFileId).toPath();
					tempFiles.add(thumbnailTempFile);

					final MediaType toMediaType = processorMediaTypes.getContainerMediaType(THUMBNAIL_TYPE)
							.withParameter("offset", getThumbnailOffset());

					final String referrer = "liferay-videoprocessor-id=" + tempFileId + ";liferay-videoprocessor-type=" + THUMBNAIL_TYPE;

					final Job job = newJob(conversionFile.get(), toMediaType, referrer);
					jobs.add(job);

					final CompletableFuture<Void> jobFuture = runJob(job, thumbnailTempFile)
							.thenRunAsync(() -> storeThumbnailImages(destinationFileVersion, thumbnailTempFile));

					jobFutures.add(jobFuture);
				}

				jobFutures.forEach(cf -> cf.exceptionally(e -> {
					LOG.error("Error executing video processor job", e);
					return null;
				}).join());
			}


		} catch (final NoSuchFileEntryException nsfee) {
			if (LOG.isDebugEnabled()) {
				LOG.debug("File entry not found", nsfee);
			}
		} finally {
			_fileVersionIds.remove(destinationFileVersion.getFileVersionId());

			final List<CompletableFuture<Void>> jobDeleteFutures = new ArrayList<>(jobs.size());
			for (final Job job : jobs) {
				final CompletableFuture<Void> jobDeleteFuture = qetcherClientService.client().deleteJob(job);
				jobDeleteFutures.add(jobDeleteFuture);
			}

			allOf(jobDeleteFutures)
			.exceptionally(e -> {
				LOG.error("Error deleting job(s)", e);
				return null;
			})
			.thenCompose(__ -> qetcherClientService.client().deleteFile(conversionFile.get()))
			.exceptionally(e -> {
				LOG.error("Error deleting file", e);
				return null;
			});

			for (final Path tempFile : tempFiles) {
				FileUtil.delete(tempFile.toFile());
			}
		}
	}

	public static <T> CompletableFuture<Void> allOf(final List<CompletableFuture<T>> cfs) {
		return CompletableFuture.allOf(cfs.toArray(new CompletableFuture[cfs.size()]));
	}

	private InputStream getContentStream(FileVersion fileVersion) {
		try {
			return fileVersion.getContentStream(false);
		} catch (final PortalException e) {
			throw new RuntimeException(e);
		}
	}

	private Job newJob(final ConversionFile conversionFile, final MediaType toMediaType, final String referrer) {
		LOG.info("Creating video processing job to {} with referrer '{}'", toMediaType, referrer);

		final InputJob inputJob = qetcherClientService.job()
				.fromFile(conversionFile)
				.fromMediaType(conversionFile.getMediaType())
				.toMediaType(toMediaType)
				.referrer(referrer)
				.build();

		return qetcherJobs.createJobWithTimeout(inputJob)
				.join();
	}

	private CompletableFuture<Path> runJob(final Job job, final Path targetFile) {
		// TODO use timeouts from configuration
		return qetcherJobs.getJobDoneWithTimeout(job)
				.thenApply(qetcherJobs::requireSuccess)
				.thenApply(qetcherJobs::getSingleResultFileId)
				.thenCompose(fileId -> qetcherClientService.client().downloadAsFile(fileId, targetFile));
	}

	private void addFileToStore(long companyId, String previewPath, String previewFilePath, Path targetFile) {
		try {
			addFileToStore(companyId, previewPath, previewFilePath, targetFile.toFile());
		} catch (final PortalException e) {
			throw new RuntimeException(e);
		}
	}

	private String getThumbnailOffset() {
		return GetterUtil.getInteger(PropsUtil.get(PropsKeys.DL_FILE_ENTRY_THUMBNAIL_VIDEO_FRAME_PERCENTAGE), 25) + "%";
	}

	private void storeThumbnailImages(FileVersion destinationFileVersion, Path thumbnailTempFile) {
		try {
			storeThumbnailImages(destinationFileVersion, thumbnailTempFile.toFile());
		} catch (final Exception e) {
			throw new RuntimeException(e);
		}
	}

}
