package de.mklinger.qetcher.liferay.client.impl.liferay71;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.liferay.dynamic.data.mapping.kernel.DDMForm;
import com.liferay.dynamic.data.mapping.kernel.DDMFormField;
import com.liferay.dynamic.data.mapping.kernel.DDMFormFieldValue;
import com.liferay.dynamic.data.mapping.kernel.DDMFormValues;
import com.liferay.dynamic.data.mapping.kernel.UnlocalizedValue;
import com.liferay.exportimport.kernel.lar.PortletDataContext;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.exception.SystemException;
import com.liferay.portal.kernel.metadata.RawMetadataProcessor;
import com.liferay.portal.kernel.repository.model.FileEntry;
import com.liferay.portal.kernel.util.LocaleUtil;
import com.liferay.portal.kernel.xml.Element;

import de.mklinger.micro.annotations.VisibleForTesting;
import de.mklinger.micro.maps.Maps;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.client.model.v1.MediaTypes;
import de.mklinger.qetcher.liferay.client.QetcherService;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.ClimateForcast;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.CreativeCommons;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.DublinCore;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.Geographic;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.HttpHeaders;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.Message;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.Office;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.OfficeOpenXMLCore;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.TIFF;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.TikaMetadataKeys;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.TikaMimeKeys;
import de.mklinger.qetcher.liferay.client.impl.liferay71.QetcherRawMetadataProcessorFields.XMPDM;
import de.mklinger.qetcher.liferay.client.impl.liferay71.scr.PropertiesFiles;

@Component(
		immediate = true,
		property = {
				"qetcher:Boolean=true",
				"service.ranking:Integer=1000"
		})
public class QetcherRawMetadataProcessorImpl implements RawMetadataProcessor {
	private static final Logger LOG = LoggerFactory.getLogger(QetcherRawMetadataProcessorImpl.class);

	private static final MediaType METADATA_FROM = MediaType.valueOf("application/octet-stream");
	private static final MediaType METADATA_TO = MediaTypes.JSON.withParameter("extractMetadata", "true");
	private static final TypeReference<HashMap<String, Object>> JSON_MAP_TYPE = new TypeReference<HashMap<String, Object>>() {};

	private static final Map<String, Field[]> fields = new HashMap<>();

	private final ObjectMapper objectMapper;
	private final Map<String, String> metadataMapping;

	@Reference
	private QetcherService qetcherService;

	public QetcherRawMetadataProcessorImpl() {
		this.objectMapper = new ObjectMapper();
		this.metadataMapping = loadMetadataMapping();
	}


	/// ---- RawMetadataProcessor API:

	@Override
	public Map<String, Field[]> getFields() {
		return fields;
	}

	@Override
	public void exportGeneratedFiles(PortletDataContext portletDataContext, FileEntry fileEntry, Element fileEntryElement) {
		// Do nothing.
	}

	@Override
	public void importGeneratedFiles(PortletDataContext portletDataContext, FileEntry fileEntry, FileEntry importedFileEntry, Element fileEntryElement) throws Exception {
		// Do nothing.
	}

	@Override
	public Map<String, DDMFormValues> getRawMetadataMap(String extension, String mimeType, File file) throws PortalException {
		try (InputStream in = new FileInputStream(file)) {
			return getRawMetadataMap(extension, mimeType, in);
		} catch (final IOException e) {
			throw new UncheckedIOException(e);
		}
	}

	@Override
	public Map<String, DDMFormValues> getRawMetadataMap(String extension, String mimeType, InputStream inputStream) throws PortalException {
		final Map<String, String> metadata = extractMetadata(inputStream, "metadata-" + extension);
		return createDDMFormValuesMap(metadata);
	}

	@VisibleForTesting
	protected Map<String, DDMFormValues> createDDMFormValuesMap(Map<String, String> metadata) {

		final Map<String, DDMFormValues> ddmFormValuesMap = new HashMap<>();

		if (metadata == null || metadata.isEmpty()) {
			return ddmFormValuesMap;
		}

		final DDMFormValues ddmFormValues = createDDMFormValues(metadata);

		final Map<String, List<DDMFormFieldValue>> ddmFormFieldsValuesMap =
				ddmFormValues.getDDMFormFieldValuesMap();

		final Set<String> names = ddmFormFieldsValuesMap.keySet();

		if (names.isEmpty()) {
			return ddmFormValuesMap;
		}

		ddmFormValuesMap.put(RawMetadataProcessor.TIKA_RAW_METADATA, ddmFormValues);

		return ddmFormValuesMap;
	}

	private DDMFormValues createDDMFormValues(Map<String, String> metadata) {

		final Locale defaultLocale = LocaleUtil.getDefault();

		final DDMForm ddmForm = createDDMForm(defaultLocale);

		final DDMFormValues ddmFormValues = new DDMFormValues(ddmForm);

		ddmFormValues.addAvailableLocale(defaultLocale);
		ddmFormValues.setDefaultLocale(defaultLocale);

		for (final String tikaName : getTikaPropertyNames()) {

			final String value = metadata.get(tikaName);
			if (value == null || value.isEmpty()) {
				continue;
			}

			final String liferayName = getLiferayDdmFieldName(tikaName);

			LOG.debug("Metadata name '{}' with value '{}'", liferayName, value);

			final DDMFormField ddmFormField = createTextDDMFormField(liferayName);

			ddmForm.addDDMFormField(ddmFormField);

			final DDMFormFieldValue ddmFormFieldValue = new DDMFormFieldValue();

			ddmFormFieldValue.setName(liferayName);
			ddmFormFieldValue.setValue(new UnlocalizedValue(value));

			ddmFormValues.addDDMFormFieldValue(ddmFormFieldValue);
		}

		return ddmFormValues;
	}

	private DDMForm createDDMForm(Locale defaultLocale) {
		final DDMForm ddmForm = new DDMForm();

		ddmForm.addAvailableLocale(defaultLocale);
		ddmForm.setDefaultLocale(defaultLocale);

		return ddmForm;
	}

	private DDMFormField createTextDDMFormField(String name) {
		final DDMFormField ddmFormField = new DDMFormField(name, "text");

		ddmFormField.setDataType("string");

		return ddmFormField;
	}

	private Collection<String> getTikaPropertyNames() {
		return metadataMapping.keySet();
	}

	private String getLiferayDdmFieldName(String tikaPropertyName) {
		return Objects.requireNonNull(metadataMapping.get(tikaPropertyName));
	}

	static {
		final List<Field> fields = new ArrayList<>();

		_addFields(ClimateForcast.class, fields);
		_addFields(CreativeCommons.class, fields);
		_addFields(DublinCore.class, fields);
		_addFields(Geographic.class, fields);
		_addFields(HttpHeaders.class, fields);
		_addFields(Message.class, fields);
		_addFields(Office.class, fields);
		_addFields(OfficeOpenXMLCore.class, fields);
		_addFields(TIFF.class, fields);
		_addFields(TikaMetadataKeys.class, fields);
		_addFields(TikaMimeKeys.class, fields);
		_addFields(XMPDM.class, fields);

		QetcherRawMetadataProcessorImpl.fields.put(
				TIKA_RAW_METADATA, fields.toArray(new Field[fields.size()]));
	}

	private static void _addFields(Class<?> clazz, List<Field> fields) {
		for (final Field field : clazz.getFields()) {
			fields.add(field);
		}
	}


	/// ---- Qetcher integration:

	private Map<String, String> loadMetadataMapping() {
		try (InputStream in = getClass().getResourceAsStream("metadata-mapping.properties")) {
			return PropertiesFiles.loadMap(in, LinkedHashMap::new);
		} catch (final IOException e) {
			throw new UncheckedIOException(e);
		}
	}

	@VisibleForTesting
	protected Map<String, String> extractMetadata(final InputStream inputStream, final String referrer) throws PortalException, SystemException {
		LOG.info("Getting metadata with referrer '{}'", referrer);

		final Map<String, Object> rawMetadata;

		try {
			rawMetadata = extractRawMetadata(
					inputStream,
					referrer);
		} catch (final IOException e) {
			throw new SystemException(e);
		}

		return toStringMap(rawMetadata);
	}

	@VisibleForTesting
	protected Map<String, String> toStringMap(final Map<String, Object> rawMetadata) {
		final HashMap<String, String> metadata = Maps.newHashMap(rawMetadata.size());

		for (final Entry<String, Object> e : rawMetadata.entrySet()) {
			final String name = e.getKey();
			final Object value = e.getValue();
			if (value instanceof String || value instanceof Number || value instanceof Boolean) {
				LOG.debug("Metadata: {} -> '{}'", name, value);
				metadata.put(name, String.valueOf(value));
			}
		}

		return metadata;
	}

	@VisibleForTesting
	protected Map<String, Object> extractRawMetadata(final InputStream inputStream, final String referrer) throws IOException {
		final byte[] json = qetcherService.convertToByteArray(
				inputStream,
				METADATA_FROM,
				METADATA_TO,
				referrer);

		return parseJson(json);
	}

	@VisibleForTesting
	protected Map<String, Object> parseJson(final byte[] json) throws IOException {
		return objectMapper.readValue(json, JSON_MAP_TYPE);
	}
}
