package de.mklinger.qetcher.liferay.client.impl.liferay71;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.document.library.kernel.document.conversion.DocumentConversion;
import com.liferay.petra.string.StringBundler;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.configuration.metatype.bnd.util.ConfigurableUtil;
import com.liferay.portal.kernel.util.FileUtil;
import com.liferay.portal.kernel.util.MimeTypesUtil;
import com.liferay.portal.kernel.util.PropsKeys;
import com.liferay.portal.kernel.util.PropsUtil;
import com.liferay.portal.kernel.util.SystemProperties;

import de.mklinger.micro.annotations.VisibleForTesting;
import de.mklinger.qetcher.client.QetcherClientException;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.liferay.client.QetcherConversionsService;
import de.mklinger.qetcher.liferay.client.QetcherService;
import de.mklinger.qetcher.liferay.client.impl.QetcherConfiguration;
import de.mklinger.qetcher.liferay.client.impl.liferay71.MultiLock.Lock;

@Component(
		immediate = true,
		property = {
				"qetcher:Boolean=true",
				"service.ranking:Integer=1000"
		},
		configurationPid = QetcherConfiguration.ID,
		configurationPolicy = ConfigurationPolicy.REQUIRE)
public class QetcherDocumentConversionImpl implements DocumentConversion {
	private static final String APPLICATION_OCTETSTREAM = "application/octet-stream";

	private static final Logger LOG = LoggerFactory.getLogger(QetcherDocumentConversionImpl.class);

	private final String[] comparableFileExtensions;
	private final String tmpDir;
	private final MultiLock conversionLock;

	private QetcherService qetcherService;
	private QetcherConversionsService qetcherConversionsService;
	private QetcherConfiguration configuration;

	public QetcherDocumentConversionImpl() {
		this(PropsUtil.getArray(PropsKeys.DL_COMPARABLE_FILE_EXTENSIONS), SystemProperties.get(SystemProperties.TMP_DIR));
	}

	@VisibleForTesting
	protected QetcherDocumentConversionImpl(String[] comparableFileExtensions, String tmpDir) {
		this.comparableFileExtensions = comparableFileExtensions;
		this.tmpDir = tmpDir;
		this.conversionLock = new MultiLock();
	}

	@Reference
	public void setQetcherService(QetcherService qetcherService) {
		this.qetcherService = qetcherService;
	}

	@Reference
	public void setQetcherConversionsService(QetcherConversionsService qetcherConversionsService) {
		this.qetcherConversionsService = qetcherConversionsService;
	}

	@Activate
	@Modified
	public synchronized void activate(final Map<String, Object> configurationProperties) {
		LOG.info("Configuration was set");
		setConfiguration(ConfigurableUtil.createConfigurable(QetcherConfiguration.class, configurationProperties));
	}

	@VisibleForTesting
	protected void setConfiguration(QetcherConfiguration configuration) {
		this.configuration = configuration;
	}

	@Override
	public File convert(String id, InputStream inputStream, String sourceExtension, String targetExtension) throws IOException {
		final Path targetFile = Paths.get(getFilePath(id, targetExtension));

		try (Lock lock = conversionLock.lock(id)) {
			if (configuration.documentConversionCacheEnabled() && Files.exists(targetFile)) {
				LOG.debug("Document conversion cache hit for id {}", id);
				return targetFile.toFile();
			} else {
				LOG.debug("Document conversion cache miss for id {}", id);
				try {
					return doConvert(id, inputStream, sourceExtension, targetExtension, targetFile);
				} catch (final Exception e) {
					FileUtil.delete(targetFile.toFile());
					throw e;
				}
			}
		}
	}

	private File doConvert(String id, InputStream inputStream, String sourceExtension, String targetExtension, final Path targetFile) throws IOException {
		final MediaType fromMediaType = getMediaTypeForExtension(sourceExtension);
		final MediaType toMediaType = getMediaTypeForExtension(targetExtension);
		final String referrer = "liferay-conversion-id=" + id;

		final Path targetDir = targetFile.getParent();
		Files.createDirectories(targetDir);

		LOG.info("Starting conversion with referrer '{}': {} -> {}", referrer, fromMediaType, toMediaType);

		qetcherService.convertToFile(inputStream, targetFile, fromMediaType, toMediaType, referrer);

		return targetFile.toFile();
	}

	public MediaType getMediaTypeForExtension(String extension) {
		// We may cache this here or in QetcherConversionsServiceImpl, but we could also cache it in our MimeTypes override.
		final String mediaTypeString = MimeTypesUtil.getExtensionContentType(extension);

		if (mediaTypeString == null || APPLICATION_OCTETSTREAM.equals(mediaTypeString)) {
			throw new QetcherClientException("Unable to map extension to media type: " + extension);
		}

		return MediaType.valueOf(mediaTypeString);
	}

	/**
	 * Get an array of target file extensions available for conversion of a file
	 * with the given source extension.
	 */
	@Override
	public String[] getConversions(String extension) {
		return qetcherConversionsService.getTargetExtensionsForExtension(extension);
	}

	// Exact copy
	@Override
	public String getFilePath(String id, String targetExtension) {
		final StringBundler sb = new StringBundler(5);

		sb.append(tmpDir);
		sb.append("/liferay/document_conversion/");
		sb.append(id);
		sb.append(StringPool.PERIOD);
		sb.append(targetExtension);

		return sb.toString();
	}

	// Exact copy
	@Override
	public boolean isComparableVersion(String extension) {
		boolean enabled = false;

		final String periodAndExtension = StringPool.PERIOD.concat(extension);

		for (final String comparableFileExtension : comparableFileExtensions) {
			if (StringPool.STAR.equals(comparableFileExtension) ||
					periodAndExtension.equals(comparableFileExtension)) {

				enabled = true;

				break;
			}
		}

		if (!enabled) {
			return false;
		}

		if (extension.equals("css") || extension.equals("htm") ||
				extension.equals("html") || extension.equals("js") ||
				extension.equals("txt") || extension.equals("xml")) {

			return true;
		}

		try {
			if (isEnabled() && isConvertBeforeCompare(extension)) {
				return true;
			}
		} catch (final Exception e) {
			LOG.error("Error in {}.isComparableVersion('{}')", getClass().getSimpleName(), extension, e);
		}

		return false;
	}

	// Exact copy
	@Override
	public boolean isConvertBeforeCompare(String extension) {
		if (extension.equals("txt")) {
			return false;
		}

		for (final String conversion : getConversions(extension)) {
			if (conversion.equals("txt")) {
				return true;
			}
		}

		return false;
	}

	@Override
	public boolean isEnabled() {
		return true;
	}

	@Override
	public void disconnect() {
		// Do nothing
	}
}
