package de.mklinger.qetcher.liferay.client.impl;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.mklinger.qetcher.client.InputJob;
import de.mklinger.qetcher.client.QetcherClientException;
import de.mklinger.qetcher.client.QetcherRemoteException;
import de.mklinger.qetcher.client.common.concurrent.Delay;
import de.mklinger.qetcher.client.model.v1.Job;
import de.mklinger.qetcher.client.model.v1.JobState;
import de.mklinger.qetcher.liferay.client.QetcherClientService;

@Component(service = QetcherJobs.class)
public class QetcherJobs {
	private static final int JOB_CREATE_TIMEOUT_SECONDS = 30;
	private static final int JOB_WAIT_TIMEOUT_MINUTES = 30;

	private static final Logger LOG = LoggerFactory.getLogger(QetcherJobs.class);

	private QetcherClientService clientService;

	@Reference
	public void setClientService(QetcherClientService clientService) {
		this.clientService = clientService;
	}

	public CompletableFuture<Job> createJobWithTimeout(final InputJob inputJob) {
		return Delay.timeout(
				clientService.client().createJob(inputJob),
				JOB_CREATE_TIMEOUT_SECONDS, TimeUnit.SECONDS,
				() -> new TimeoutException(
						"Timeout creating job: "
								+ inputJob.getFromMediaType()
								+ " -> "
								+ inputJob.getToMediaType()));
	}

	public CompletableFuture<Job> getJobDoneWithTimeout(final Job job) {
		return Delay.timeout(
				clientService.client().getJobDone(job),
				JOB_WAIT_TIMEOUT_MINUTES, TimeUnit.MINUTES,
				() -> new TimeoutException(
						"Timeout getting job done "
								+ job.getJobId()
								+ " "
								+ job.getFromMediaType()
								+ " -> "
								+ job.getToMediaType()));
	}

	public Job requireSuccess(final Job job) {
		if (job.getState() != JobState.SUCCESS) {
			Exception cause;
			if (job.getError() != null) {
				cause = new QetcherRemoteException(job.getError(), -1);
			} else {
				cause = null;
			}
			throw new QetcherClientException("Job " + job.getJobId() + " is not in state SUCCESS but " + job.getState(), cause);
		}

		return job;
	}

	public CompletableFuture<Void> deleteJobWithTimeout(final Job job) {
		return Delay.timeout(
				clientService.client().deleteJob(job),
				30, TimeUnit.SECONDS,
				() -> new TimeoutException(
						"Timeout deleting job "
								+ job.getJobId()
								+ " "
								+ job.getFromMediaType()
								+ " -> "
								+ job.getToMediaType()));
	}

	public String getSingleResultFileId(final Job job) {
		final List<String> resultFileIds = job.getResultFileIds();
		if (resultFileIds == null || resultFileIds.isEmpty()) {
			throw new QetcherClientException("Expected 1 result file but have no result files");
		}
		if (resultFileIds.size() != 1) {
			LOG.warn("Expected 1 result file but is " + resultFileIds.size());
		}
		return resultFileIds.get(0);
	}
}
