package de.mklinger.qetcher.liferay.client.impl.liferay71.scr;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Timer;
import java.util.TimerTask;
import java.util.TreeMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.Filter;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.service.component.runtime.ServiceComponentRuntime;
import org.osgi.service.component.runtime.dto.ComponentDescriptionDTO;
import org.osgi.service.component.runtime.dto.ReferenceDTO;
import org.osgi.util.tracker.ServiceTracker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.portal.kernel.util.PropsKeys;
import com.liferay.portal.kernel.util.PropsUtil;

import de.mklinger.micro.annotations.GuardedBy;

public class ScrServiceOverride<T> implements AutoCloseable {
	// See https://portal.liferay.dev/docs/7-0/tutorials/-/knowledge_base/t/overriding-service-references

	private static final Logger LOG = LoggerFactory.getLogger(ScrServiceOverride.class);

	private final BundleContext bundleContext;
	private final Class<T> serviceClass;
	private final String appliedTarget;

	// default rule:
	private final Predicate<ServiceReference<T>> shouldApplyOverride;

	private final Consumer<ServiceReference<T>> onOverridingServiceAvailable;
	private final Runnable onOverridingServiceUnavailable;

	private final ServiceTracker<ServiceComponentRuntime, ServiceComponentRuntime> scrTracker;
	private final ServiceAvailableTracker<T> overridingServiceTracker;

	private final Timer timer;
	@GuardedBy("timer")
	private TimerTask timerTask;

	private final List<Path> managedConfigFiles = new CopyOnWriteArrayList<>();

	public ScrServiceOverride(BundleContext bundleContext, Class<T> serviceClass, String filterKey, String filterValue) {
		this(bundleContext, serviceClass, filterKey, filterValue, null, null);
	}

	public ScrServiceOverride(BundleContext bundleContext,
			Class<T> serviceClass, String filterKey, String filterValue,
			Consumer<ServiceReference<T>> onOverridingServiceAvailable,
			Runnable onOverridingServiceUnavailable) {

		this.bundleContext = bundleContext;
		this.serviceClass = serviceClass;
		this.appliedTarget = FilterFactory.eq(filterKey, filterValue);
		this.shouldApplyOverride = sr -> !filterValue.equals(sr.getProperty(filterKey));

		this.onOverridingServiceAvailable = onOverridingServiceAvailable;
		this.onOverridingServiceUnavailable = onOverridingServiceUnavailable;

		this.timer = new Timer();

		this.scrTracker = new ServiceTracker<>(bundleContext, ServiceComponentRuntime.class, null);
		this.scrTracker.open();

		final Filter filter = FilterFactory.build(
				FilterFactory.and(
						FilterFactory.eq(Constants.OBJECTCLASS, serviceClass.getName()),
						FilterFactory.eq(filterKey, filterValue)
						)
				);
		this.overridingServiceTracker = new ServiceAvailableTracker<>(
				bundleContext,
				filter,
				this::onOverridingServiceAvailable,
				this::onOverridingServiceUnavailable);
		this.overridingServiceTracker.open(true);
	}

	private void onOverridingServiceAvailable(ServiceReference<T> serviceReference) {
		if (onOverridingServiceAvailable != null) {
			onOverridingServiceAvailable.accept(serviceReference);
		}
		start();
	}

	private void onOverridingServiceUnavailable() {
		if (onOverridingServiceUnavailable != null) {
			onOverridingServiceUnavailable.run();
		}
		stop();
	}

	private void start() {
		scheduleTimerAtFixedRate(this::apply, 0, 30, TimeUnit.SECONDS);
	}

	private void stop() {
		cancelTimerTask();
	}

	private void apply() {
		final ServiceComponentRuntime serviceComponentRuntime = scrTracker.getService();
		if (serviceComponentRuntime == null) {
			LOG.warn("Service component runtime not available");
			return;
		}

		getAllServiceReferencesForServiceClass().stream()
		.filter(shouldApplyOverride)
		.peek(serviceReference -> LOG.debug("Considering service reference {}", serviceReference))
		.map(ServiceReference::getUsingBundles)
		.filter(Objects::nonNull)
		.flatMap(Stream::of)
		.distinct()
		.peek(usingBundle -> LOG.debug("Considering bundle {}", usingBundle))
		.map(serviceComponentRuntime::getComponentDescriptionDTOs)
		.flatMap(Collection::stream)
		.peek(componentDescriptionDTO -> LOG.debug("Considering component {}", componentDescriptionDTO.name))
		.map(componentDescriptionDTO -> MatchingReferences.map(componentDescriptionDTO, this::isMatchingReference))
		.filter(Optional::isPresent)
		.map(Optional::get)
		.forEach(this::applyComponentConfigurationIfNeeded);
	}

	private void applyComponentConfigurationIfNeeded(final MatchingReferences matchingReferences) {
		final String configFilename = matchingReferences.getComponentName() + ".cfg"; // cfg is properties format
		final Path configPath = Paths.get(PropsUtil.get(PropsKeys.LIFERAY_HOME), "osgi", "configs", configFilename);

		final TreeMap<String, String> properties = new TreeMap<>();
		for (final String referenceName : matchingReferences.getReferenceNames()) {
			properties.put(referenceName + ".target", appliedTarget);
		}

		if (LOG.isDebugEnabled()) {
			LOG.debug("********************");
			LOG.debug("* {} Override:", serviceClass.getSimpleName());
			LOG.debug("* Found SCR reference for service that should be overridden:");
			LOG.debug("* To override, use config file:");
			LOG.debug("* {}", configPath);
			LOG.debug("* with properties");
			LOG.debug("* {}", properties);
			LOG.debug("********************");
		}

		try {
			applyConfigProperties(configPath, properties);
		} catch (final IOException e) {
			throw new UncheckedIOException(e);
		}
	}

	private void applyConfigProperties(final Path configPath, final TreeMap<String, String> properties) throws IOException {
		final boolean store;

		if (Files.exists(configPath)) {
			if (managedConfigFiles.contains(configPath)) {
				if (hasExpectedContents(configPath, properties)) {
					LOG.debug("{} override: Managed config file exists and has expected contents. Nothing to be done: {}", serviceClass.getSimpleName(), configPath);
					store = false;
				} else {
					LOG.info("{} override: Updating managed config file: {}", serviceClass.getSimpleName(), configPath);
					store = true;
				}
			} else {
				LOG.warn("{} override: Config file already exists, unable to override service: {}", serviceClass.getSimpleName(), configPath);
				store = false;
			}
		} else {
			LOG.info("{} override: Creating new managed config file: {}", serviceClass.getSimpleName(), configPath);
			store = true;
		}

		if (store) {
			PropertiesFiles.store(properties, configPath);
			managedConfigFiles.add(configPath);
		}
	}

	private boolean hasExpectedContents(Path configPath, TreeMap<String, String> expectedProperties) throws IOException {
		final TreeMap<String, String> actualProperties = PropertiesFiles.loadMap(configPath, TreeMap::new);
		return actualProperties.equals(expectedProperties);
	}

	private static class MatchingReferences {
		private final ComponentDescriptionDTO component;
		private final List<ReferenceDTO> references;

		public MatchingReferences(ComponentDescriptionDTO component, List<ReferenceDTO> references) {
			this.component = component;
			this.references = references;
		}

		public static Optional<MatchingReferences> map(ComponentDescriptionDTO componentDescriptionDTO, Predicate<ReferenceDTO> predicate) {
			final List<ReferenceDTO> matchingReferences = Stream.of(componentDescriptionDTO.references)
					.filter(predicate)
					.collect(Collectors.toList());

			if (matchingReferences.isEmpty()) {
				return Optional.empty();
			} else {
				return Optional.of(new MatchingReferences(componentDescriptionDTO, matchingReferences));
			}
		}

		public String getComponentName() {
			return component.name;
		}

		public List<String> getReferenceNames() {
			return references.stream()
					.map(reference -> reference.name)
					.collect(Collectors.toList());
		}
	}

	private boolean isMatchingReference(ReferenceDTO reference) {
		if (!serviceClass.getName().equals(reference.interfaceName)) {
			return false;
		} else if (reference.target == null) {
			return true;
		} else if (appliedTarget.equals(reference.target)) { // TODO normalize filters?
			// Already applied
			return false;
		} else if (FilterFactory.not(appliedTarget).equals(reference.target)) { // TODO normalize filters?
			LOG.debug("{} override: Not overriding service reference {} that explicitly negates the override target: {}", serviceClass.getSimpleName(), reference.name, reference.target);
			return false;
		} else {
			LOG.warn("{} override: Not overriding service reference {} with target '{}'", serviceClass.getSimpleName(), reference.name, reference.target);
			return false;
		}
	}

	private Collection<ServiceReference<T>> getAllServiceReferencesForServiceClass() {
		try {
			return bundleContext.getServiceReferences(serviceClass, null);
		} catch (final InvalidSyntaxException e) {
			throw new AssertionError(e);
		}
	}

	@Override
	public void close() {
		overridingServiceTracker.close();
		stop();
		scrTracker.close();
		timer.cancel();

		for (final Path configFile : managedConfigFiles) {
			try {
				LOG.info("{} override: Deleting managed config file: {}", serviceClass.getSimpleName(), configFile);
				Files.delete(configFile);
			} catch (final IOException e) {
				throw new UncheckedIOException(e);
			}
		}
	}

	private void cancelTimerTask() {
		synchronized (timer) {
			if (timerTask != null) {
				timerTask.cancel();
				timerTask = null;
			}
		}
	}

	private void scheduleTimerAtFixedRate(Runnable r,
			long initialDelay,
			long period,
			TimeUnit unit) {
		synchronized (timer) {
			cancelTimerTask();
			timerTask = new TimerTask() {
				@Override
				public void run() {
					r.run();
				}
			};
			timer.scheduleAtFixedRate(timerTask, unit.toMillis(initialDelay), unit.toMillis(period));
		}
	}
}
