package de.mklinger.qetcher.liferay.client.impl.liferay71.scr;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

/**
 * Read and write support for properties files with {@link Map}s and functional style.
 *
 * Files are compatible with {@link java.util.Properties} generated files.
 *
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class PropertiesFiles {
	private PropertiesFiles() {}

	public static HashMap<String, String> loadHashMap(final Reader reader) throws IOException {
		return loadMap(reader, HashMap::new);
	}

	public static HashMap<String, String> loadHashMap(final InputStream in) throws IOException {
		return loadMap(in, HashMap::new);
	}

	public static HashMap<String, String> loadHashMap(final Path file) throws IOException {
		return loadMap(file, HashMap::new);
	}

	public static <T extends Map<String, String>> T loadMap(Reader reader, Supplier<T> mapSupplier) throws IOException {
		final T map = mapSupplier.get();
		load(reader, map::put);
		return map;
	}

	public static <T extends Map<String, String>> T loadMap(InputStream in, Supplier<T> mapSupplier) throws IOException {
		final T map = mapSupplier.get();
		load(in, map::put);
		return map;
	}

	public static <T extends Map<String, String>> T loadMap(Path file, Supplier<T> mapSupplier) throws IOException {
		final T map = mapSupplier.get();
		try (InputStream in = Files.newInputStream(file)) {
			load(in, map::put);
		}
		return map;
	}

	public static void store(final Map<String, String> map, final Writer writer) throws IOException {
		store(map.entrySet(), writer, null);
	}

	public static void store(final Map<String, String> map, final Writer writer, final String comments) throws IOException {
		store(map.entrySet(), writer, comments);
	}

	public static void store(final Map<String, String> map, final OutputStream out) throws IOException {
		store(map.entrySet(), out, null);
	}

	public static void store(final Map<String, String> map, final OutputStream out, final String comments) throws IOException {
		store(map.entrySet(), out, comments);
	}

	public static void store(final Map<String, String> map, final Path file) throws IOException {
		try (OutputStream out = Files.newOutputStream(file)) {
			store(map.entrySet(), out, null);
		}
	}

	public static void store(final Map<String, String> map, final Path file, final String comments) throws IOException {
		try (OutputStream out = Files.newOutputStream(file)) {
			store(map.entrySet(), out, comments);
		}
	}

	/**
	 * Reads a property list (key and element pairs) from the input
	 * character stream in a simple line-oriented format.
	 * <p>
	 * Properties are processed in terms of lines. There are two
	 * kinds of line, <i>natural lines</i> and <i>logical lines</i>.
	 * A natural line is defined as a line of
	 * characters that is terminated either by a set of line terminator
	 * characters ({@code \n} or {@code \r} or {@code \r\n})
	 * or by the end of the stream. A natural line may be either a blank line,
	 * a comment line, or hold all or some of a key-element pair. A logical
	 * line holds all the data of a key-element pair, which may be spread
	 * out across several adjacent natural lines by escaping
	 * the line terminator sequence with a backslash character
	 * {@code \}.  Note that a comment line cannot be extended
	 * in this manner; every natural line that is a comment must have
	 * its own comment indicator, as described below. Lines are read from
	 * input until the end of the stream is reached.
	 *
	 * <p>
	 * A natural line that contains only white space characters is
	 * considered blank and is ignored.  A comment line has an ASCII
	 * {@code '#'} or {@code '!'} as its first non-white
	 * space character; comment lines are also ignored and do not
	 * encode key-element information.  In addition to line
	 * terminators, this format considers the characters space
	 * ({@code ' '}, {@code '\u005Cu0020'}), tab
	 * ({@code '\t'}, {@code '\u005Cu0009'}), and form feed
	 * ({@code '\f'}, {@code '\u005Cu000C'}) to be white
	 * space.
	 *
	 * <p>
	 * If a logical line is spread across several natural lines, the
	 * backslash escaping the line terminator sequence, the line
	 * terminator sequence, and any white space at the start of the
	 * following line have no affect on the key or element values.
	 * The remainder of the discussion of key and element parsing
	 * (when loading) will assume all the characters constituting
	 * the key and element appear on a single natural line after
	 * line continuation characters have been removed.  Note that
	 * it is <i>not</i> sufficient to only examine the character
	 * preceding a line terminator sequence to decide if the line
	 * terminator is escaped; there must be an odd number of
	 * contiguous backslashes for the line terminator to be escaped.
	 * Since the input is processed from left to right, a
	 * non-zero even number of 2<i>n</i> contiguous backslashes
	 * before a line terminator (or elsewhere) encodes <i>n</i>
	 * backslashes after escape processing.
	 *
	 * <p>
	 * The key contains all of the characters in the line starting
	 * with the first non-white space character and up to, but not
	 * including, the first unescaped {@code '='},
	 * {@code ':'}, or white space character other than a line
	 * terminator. All of these key termination characters may be
	 * included in the key by escaping them with a preceding backslash
	 * character; for example,<p>
	 *
	 * {@code \:\=}<p>
	 *
	 * would be the two-character key {@code ":="}.  Line
	 * terminator characters can be included using {@code \r} and
	 * {@code \n} escape sequences.  Any white space after the
	 * key is skipped; if the first non-white space character after
	 * the key is {@code '='} or {@code ':'}, then it is
	 * ignored and any white space characters after it are also
	 * skipped.  All remaining characters on the line become part of
	 * the associated element string; if there are no remaining
	 * characters, the element is the empty string
	 * {@code ""}.  Once the raw character sequences
	 * constituting the key and element are identified, escape
	 * processing is performed as described above.
	 *
	 * <p>
	 * As an example, each of the following three lines specifies the key
	 * {@code "Truth"} and the associated element value
	 * {@code "Beauty"}:
	 * <pre>
	 * Truth = Beauty
	 *  Truth:Beauty
	 * Truth                    :Beauty
	 * </pre>
	 * As another example, the following three lines specify a single
	 * property:
	 * <pre>
	 * fruits                           apple, banana, pear, \
	 *                                  cantaloupe, watermelon, \
	 *                                  kiwi, mango
	 * </pre>
	 * The key is {@code "fruits"} and the associated element is:
	 * <pre>"apple, banana, pear, cantaloupe, watermelon, kiwi, mango"</pre>
	 * Note that a space appears before each {@code \} so that a space
	 * will appear after each comma in the final result; the {@code \},
	 * line terminator, and leading white space on the continuation line are
	 * merely discarded and are <i>not</i> replaced by one or more other
	 * characters.
	 * <p>
	 * As a third example, the line:
	 * <pre>cheeses
	 * </pre>
	 * specifies that the key is {@code "cheeses"} and the associated
	 * element is the empty string {@code ""}.
	 * <p>
	 * <a id="unicodeescapes"></a>
	 * Characters in keys and elements can be represented in escape
	 * sequences similar to those used for character and string literals
	 * (see sections 3.3 and 3.10.6 of
	 * <cite>The Java&trade; Language Specification</cite>).
	 *
	 * The differences from the character escape sequences and Unicode
	 * escapes used for characters and strings are:
	 *
	 * <ul>
	 * <li> Octal escapes are not recognized.
	 *
	 * <li> The character sequence {@code \b} does <i>not</i>
	 * represent a backspace character.
	 *
	 * <li> The method does not treat a backslash character,
	 * {@code \}, before a non-valid escape character as an
	 * error; the backslash is silently dropped.  For example, in a
	 * Java string the sequence {@code "\z"} would cause a
	 * compile time error.  In contrast, this method silently drops
	 * the backslash.  Therefore, this method treats the two character
	 * sequence {@code "\b"} as equivalent to the single
	 * character {@code 'b'}.
	 *
	 * <li> Escapes are not necessary for single and double quotes;
	 * however, by the rule above, single and double quote characters
	 * preceded by a backslash still yield single and double quote
	 * characters, respectively.
	 *
	 * <li> Only a single 'u' character is allowed in a Unicode escape
	 * sequence.
	 *
	 * </ul>
	 * <p>
	 * The specified stream remains open after this method returns.
	 *
	 * @param   reader   the input character stream.
	 * @throws  IOException  if an error occurred when reading from the
	 *          input stream.
	 * @throws  IllegalArgumentException if a malformed Unicode escape
	 *          appears in the input.
	 * @throws  NullPointerException if {@code reader} is null.
	 * @since   1.6
	 */
	public static void load(final Reader reader, final BiConsumer<String, String> consumer) throws IOException {
		Objects.requireNonNull(reader, "reader parameter is null");
		Objects.requireNonNull(consumer, "consumer parameter is null");
		load0(new LineReader(reader), consumer);
	}

	/**
	 * Reads a property list (key and element pairs) from the input
	 * byte stream. The input stream is in a simple line-oriented
	 * format as specified in
	 * {@link #load(java.io.Reader) load(Reader)} and is assumed to use
	 * the ISO 8859-1 character encoding; that is each byte is one Latin1
	 * character. Characters not in Latin1, and certain special characters,
	 * are represented in keys and elements using Unicode escapes as defined in
	 * section 3.3 of
	 * <cite>The Java&trade; Language Specification</cite>.
	 * <p>
	 * The specified stream remains open after this method returns.
	 *
	 * @param      inStream   the input stream.
	 * @exception  IOException  if an error occurred when reading from the
	 *             input stream.
	 * @throws     IllegalArgumentException if the input stream contains a
	 *             malformed Unicode escape sequence.
	 * @throws     NullPointerException if {@code inStream} is null.
	 * @since 1.2
	 */
	public static void load(final InputStream inStream, final BiConsumer<String, String> consumer) throws IOException {
		Objects.requireNonNull(inStream, "inStream parameter is null");
		Objects.requireNonNull(consumer, "consumer parameter is null");
		load0(new LineReader(inStream), consumer);
	}

	private static void load0 (final LineReader lr, final BiConsumer<String, String> consumer) throws IOException {
		final char[] convtBuf = new char[1024];
		int limit;
		int keyLen;
		int valueStart;
		char c;
		boolean hasSep;
		boolean precedingBackslash;

		while ((limit = lr.readLine()) >= 0) {
			c = 0;
			keyLen = 0;
			valueStart = limit;
			hasSep = false;

			//System.out.println("line=<" + new String(lineBuf, 0, limit) + ">");
			precedingBackslash = false;
			while (keyLen < limit) {
				c = lr.lineBuf[keyLen];
				//need check if escaped.
				if ((c == '=' ||  c == ':') && !precedingBackslash) {
					valueStart = keyLen + 1;
					hasSep = true;
					break;
				} else if ((c == ' ' || c == '\t' ||  c == '\f') && !precedingBackslash) {
					valueStart = keyLen + 1;
					break;
				}
				if (c == '\\') {
					precedingBackslash = !precedingBackslash;
				} else {
					precedingBackslash = false;
				}
				keyLen++;
			}
			while (valueStart < limit) {
				c = lr.lineBuf[valueStart];
				if (c != ' ' && c != '\t' &&  c != '\f') {
					if (!hasSep && (c == '=' ||  c == ':')) {
						hasSep = true;
					} else {
						break;
					}
				}
				valueStart++;
			}
			final String key = loadConvert(lr.lineBuf, 0, keyLen, convtBuf);
			final String value = loadConvert(lr.lineBuf, valueStart, limit - valueStart, convtBuf);
			consumer.accept(key, value);
		}
	}

	/* Read in a "logical line" from an InputStream/Reader, skip all comment
	 * and blank lines and filter out those leading whitespace characters
	 * (\u0020, \u0009 and \u000c) from the beginning of a "natural line".
	 * Method returns the char length of the "logical line" and stores
	 * the line in "lineBuf".
	 */
	private static class LineReader {
		public LineReader(final InputStream inStream) {
			this.inStream = inStream;
			inByteBuf = new byte[8192];
		}

		public LineReader(final Reader reader) {
			this.reader = reader;
			inCharBuf = new char[8192];
		}

		byte[] inByteBuf;
		char[] inCharBuf;
		char[] lineBuf = new char[1024];
		int inLimit = 0;
		int inOff = 0;
		InputStream inStream;
		Reader reader;

		int readLine() throws IOException {
			int len = 0;
			char c = 0;

			boolean skipWhiteSpace = true;
			boolean isCommentLine = false;
			boolean isNewLine = true;
			boolean appendedLineBegin = false;
			boolean precedingBackslash = false;
			boolean skipLF = false;

			while (true) {
				if (inOff >= inLimit) {
					inLimit = (inStream==null)?reader.read(inCharBuf)
							:inStream.read(inByteBuf);
					inOff = 0;
					if (inLimit <= 0) {
						if (len == 0 || isCommentLine) {
							return -1;
						}
						if (precedingBackslash) {
							len--;
						}
						return len;
					}
				}
				if (inStream != null) {
					//The line below is equivalent to calling a
					//ISO8859-1 decoder.
					c = (char)(inByteBuf[inOff++] & 0xFF);
				} else {
					c = inCharBuf[inOff++];
				}
				if (skipLF) {
					skipLF = false;
					if (c == '\n') {
						continue;
					}
				}
				if (skipWhiteSpace) {
					if (c == ' ' || c == '\t' || c == '\f') {
						continue;
					}
					if (!appendedLineBegin && (c == '\r' || c == '\n')) {
						continue;
					}
					skipWhiteSpace = false;
					appendedLineBegin = false;
				}
				if (isNewLine) {
					isNewLine = false;
					if (c == '#' || c == '!') {
						// Comment, quickly consume the rest of the line,
						// resume on line-break and backslash.
						if (inStream != null) {
							while (inOff < inLimit) {
								final byte b = inByteBuf[inOff++];
								if (b == '\n' || b == '\r' || b == '\\') {
									c = (char)(b & 0xFF);
									break;
								}
							}
						} else {
							while (inOff < inLimit) {
								c = inCharBuf[inOff++];
								if (c == '\n' || c == '\r' || c == '\\') {
									break;
								}
							}
						}
						isCommentLine = true;
					}
				}

				if (c != '\n' && c != '\r') {
					lineBuf[len++] = c;
					if (len == lineBuf.length) {
						int newLength = lineBuf.length * 2;
						if (newLength < 0) {
							newLength = Integer.MAX_VALUE;
						}
						final char[] buf = new char[newLength];
						System.arraycopy(lineBuf, 0, buf, 0, lineBuf.length);
						lineBuf = buf;
					}
					//flip the preceding backslash flag
					if (c == '\\') {
						precedingBackslash = !precedingBackslash;
					} else {
						precedingBackslash = false;
					}
				}
				else {
					// reached EOL
					if (isCommentLine || len == 0) {
						isCommentLine = false;
						isNewLine = true;
						skipWhiteSpace = true;
						len = 0;
						continue;
					}
					if (inOff >= inLimit) {
						inLimit = (inStream==null)
								?reader.read(inCharBuf)
										:inStream.read(inByteBuf);
								inOff = 0;
								if (inLimit <= 0) {
									if (precedingBackslash) {
										len--;
									}
									return len;
								}
					}
					if (precedingBackslash) {
						len -= 1;
						//skip the leading whitespace characters in following line
						skipWhiteSpace = true;
						appendedLineBegin = true;
						precedingBackslash = false;
						if (c == '\r') {
							skipLF = true;
						}
					} else {
						return len;
					}
				}
			}
		}
	}

	/*
	 * Converts encoded &#92;uxxxx to unicode chars
	 * and changes special saved chars to their original forms
	 */
	private static String loadConvert (final char[] in, int off, final int len, char[] convtBuf) {
		if (convtBuf.length < len) {
			int newLen = len * 2;
			if (newLen < 0) {
				newLen = Integer.MAX_VALUE;
			}
			convtBuf = new char[newLen];
		}
		char aChar;
		final char[] out = convtBuf;
		int outLen = 0;
		final int end = off + len;

		while (off < end) {
			aChar = in[off++];
			if (aChar == '\\') {
				aChar = in[off++];
				if(aChar == 'u') {
					// Read the xxxx
					int value=0;
					for (int i=0; i<4; i++) {
						aChar = in[off++];
						switch (aChar) {
						case '0': case '1': case '2': case '3': case '4':
						case '5': case '6': case '7': case '8': case '9':
							value = (value << 4) + aChar - '0';
							break;
						case 'a': case 'b': case 'c':
						case 'd': case 'e': case 'f':
							value = (value << 4) + 10 + aChar - 'a';
							break;
						case 'A': case 'B': case 'C':
						case 'D': case 'E': case 'F':
							value = (value << 4) + 10 + aChar - 'A';
							break;
						default:
							throw new IllegalArgumentException(
									"Malformed \\uxxxx encoding.");
						}
					}
					out[outLen++] = (char)value;
				} else {
					if (aChar == 't') {
						aChar = '\t';
					} else if (aChar == 'r') {
						aChar = '\r';
					} else if (aChar == 'n') {
						aChar = '\n';
					} else if (aChar == 'f') {
						aChar = '\f';
					}
					out[outLen++] = aChar;
				}
			} else {
				out[outLen++] = aChar;
			}
		}
		return new String (out, 0, outLen);
	}




	/**
	 * Writes this property list (key and element pairs) in this
	 * {@code Properties} table to the output character stream in a
	 * format suitable for using the {@link #load(java.io.Reader) load(Reader)}
	 * method.
	 * <p>
	 * Properties from the defaults table of this {@code Properties}
	 * table (if any) are <i>not</i> written out by this method.
	 * <p>
	 * If the comments argument is not null, then an ASCII {@code #}
	 * character, the comments string, and a line separator are first written
	 * to the output stream. Thus, the {@code comments} can serve as an
	 * identifying comment. Any one of a line feed ('\n'), a carriage
	 * return ('\r'), or a carriage return followed immediately by a line feed
	 * in comments is replaced by a line separator generated by the {@code Writer}
	 * and if the next character in comments is not character {@code #} or
	 * character {@code !} then an ASCII {@code #} is written out
	 * after that line separator.
	 * <p>
	 * Next, a comment line is always written, consisting of an ASCII
	 * {@code #} character, the current date and time (as if produced
	 * by the {@code toString} method of {@code Date} for the
	 * current time), and a line separator as generated by the {@code Writer}.
	 * <p>
	 * Then every entry in this {@code Properties} table is
	 * written out, one per line. For each entry the key string is
	 * written, then an ASCII {@code =}, then the associated
	 * element string. For the key, all space characters are
	 * written with a preceding {@code \} character.  For the
	 * element, leading space characters, but not embedded or trailing
	 * space characters, are written with a preceding {@code \}
	 * character. The key and element characters {@code #},
	 * {@code !}, {@code =}, and {@code :} are written
	 * with a preceding backslash to ensure that they are properly loaded.
	 * <p>
	 * After the entries have been written, the output stream is flushed.
	 * The output stream remains open after this method returns.
	 *
	 * @param   writer      an output character stream writer.
	 * @param   comments   a description of the property list.
	 * @exception  IOException if writing this property list to the specified
	 *             output stream throws an {@code IOException}.
	 * @exception  ClassCastException  if this {@code Properties} object
	 *             contains any keys or values that are not {@code Strings}.
	 * @exception  NullPointerException  if {@code writer} is null.
	 * @since 1.6
	 */
	public static void store(final Iterable<Map.Entry<String, String>> entrySet, final Writer writer, final String comments)
			throws IOException
	{
		Objects.requireNonNull(entrySet, "entrySetSupplier parameter is null");
		store0(entrySet,
				(writer instanceof BufferedWriter)?(BufferedWriter)writer
						: new BufferedWriter(writer),
						comments,
						false);
	}

	/**
	 * Writes this property list (key and element pairs) in this
	 * {@code Properties} table to the output stream in a format suitable
	 * for loading into a {@code Properties} table using the
	 * {@link #load(InputStream) load(InputStream)} method.
	 * <p>
	 * Properties from the defaults table of this {@code Properties}
	 * table (if any) are <i>not</i> written out by this method.
	 * <p>
	 * This method outputs the comments, properties keys and values in
	 * the same format as specified in
	 * {@link #store(java.io.Writer, java.lang.String) store(Writer)},
	 * with the following differences:
	 * <ul>
	 * <li>The stream is written using the ISO 8859-1 character encoding.
	 *
	 * <li>Characters not in Latin-1 in the comments are written as
	 * {@code \u005Cu}<i>xxxx</i> for their appropriate unicode
	 * hexadecimal value <i>xxxx</i>.
	 *
	 * <li>Characters less than {@code \u005Cu0020} and characters greater
	 * than {@code \u005Cu007E} in property keys or values are written
	 * as {@code \u005Cu}<i>xxxx</i> for the appropriate hexadecimal
	 * value <i>xxxx</i>.
	 * </ul>
	 * <p>
	 * After the entries have been written, the output stream is flushed.
	 * The output stream remains open after this method returns.
	 *
	 * @param   out      an output stream.
	 * @param   comments   a description of the property list.
	 * @exception  IOException if writing this property list to the specified
	 *             output stream throws an {@code IOException}.
	 * @exception  ClassCastException  if this {@code Properties} object
	 *             contains any keys or values that are not {@code Strings}.
	 * @exception  NullPointerException  if {@code out} is null.
	 * @since 1.2
	 */
	public static void store(final Iterable<Map.Entry<String, String>> entrySet, final OutputStream out, final String comments)
			throws IOException
	{
		Objects.requireNonNull(entrySet, "entrySet parameter is null");
		store0(entrySet,
				new BufferedWriter(new OutputStreamWriter(out, "8859_1")),
				comments,
				true);
	}

	private static void store0(final Iterable<Map.Entry<String, String>> entrySet, final BufferedWriter bw, final String comments, final boolean escUnicode)
			throws IOException
	{
		if (comments != null) {
			writeComments(bw, comments);
		}
		bw.write("#" + new Date().toString());
		bw.newLine();
		for (final Map.Entry<String, String> e : entrySet) {
			String key = e.getKey();
			String val = e.getValue();
			key = saveConvert(key, true, escUnicode);
			/* No need to escape embedded and trailing spaces for value, hence
			 * pass false to flag.
			 */
			val = saveConvert(val, false, escUnicode);
			bw.write(key + "=" + val);
			bw.newLine();
		}
		bw.flush();
	}

	private static void writeComments(final BufferedWriter bw, final String comments)
			throws IOException {
		bw.write("#");
		final int len = comments.length();
		int current = 0;
		int last = 0;
		final char[] uu = new char[6];
		uu[0] = '\\';
		uu[1] = 'u';
		while (current < len) {
			final char c = comments.charAt(current);
			if (c > '\u00ff' || c == '\n' || c == '\r') {
				if (last != current) {
					bw.write(comments.substring(last, current));
				}
				if (c > '\u00ff') {
					uu[2] = toHex((c >> 12) & 0xf);
					uu[3] = toHex((c >>  8) & 0xf);
					uu[4] = toHex((c >>  4) & 0xf);
					uu[5] = toHex( c        & 0xf);
					bw.write(new String(uu));
				} else {
					bw.newLine();
					if (c == '\r' &&
							current != len - 1 &&
							comments.charAt(current + 1) == '\n') {
						current++;
					}
					if (current == len - 1 ||
							(comments.charAt(current + 1) != '#' &&
							comments.charAt(current + 1) != '!')) {
						bw.write("#");
					}
				}
				last = current + 1;
			}
			current++;
		}
		if (last != current) {
			bw.write(comments.substring(last, current));
		}
		bw.newLine();
	}

	/*
	 * Converts unicodes to encoded &#92;uxxxx and escapes
	 * special characters with a preceding slash
	 */
	private static String saveConvert(final String theString,
			final boolean escapeSpace,
			final boolean escapeUnicode) {
		final int len = theString.length();
		int bufLen = len * 2;
		if (bufLen < 0) {
			bufLen = Integer.MAX_VALUE;
		}
		final StringBuilder outBuffer = new StringBuilder(bufLen);

		for(int x=0; x<len; x++) {
			final char aChar = theString.charAt(x);
			// Handle common case first, selecting largest block that
			// avoids the specials below
			if ((aChar > 61) && (aChar < 127)) {
				if (aChar == '\\') {
					outBuffer.append('\\'); outBuffer.append('\\');
					continue;
				}
				outBuffer.append(aChar);
				continue;
			}
			switch(aChar) {
			case ' ':
				if (x == 0 || escapeSpace) {
					outBuffer.append('\\');
				}
				outBuffer.append(' ');
				break;
			case '\t':outBuffer.append('\\'); outBuffer.append('t');
			break;
			case '\n':outBuffer.append('\\'); outBuffer.append('n');
			break;
			case '\r':outBuffer.append('\\'); outBuffer.append('r');
			break;
			case '\f':outBuffer.append('\\'); outBuffer.append('f');
			break;
			case '=': // Fall through
			case ':': // Fall through
			case '#': // Fall through
			case '!':
				outBuffer.append('\\'); outBuffer.append(aChar);
				break;
			default:
				if (((aChar < 0x0020) || (aChar > 0x007e)) & escapeUnicode ) {
					outBuffer.append('\\');
					outBuffer.append('u');
					outBuffer.append(toHex((aChar >> 12) & 0xF));
					outBuffer.append(toHex((aChar >>  8) & 0xF));
					outBuffer.append(toHex((aChar >>  4) & 0xF));
					outBuffer.append(toHex( aChar        & 0xF));
				} else {
					outBuffer.append(aChar);
				}
			}
		}
		return outBuffer.toString();
	}

	/**
	 * Convert a nibble to a hex character
	 * @param   nibble  the nibble to convert.
	 */
	private static char toHex(final int nibble) {
		return hexDigit[(nibble & 0xF)];
	}

	/** A table of hex digits */
	private static final char[] hexDigit = {
			'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'
	};
}
