/*
 * Copyright 2013-present mklinger GmbH - http://www.mklinger.de
 *
 * All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of mklinger GmbH and its suppliers, if any.
 * The intellectual and technical concepts contained herein are
 * proprietary to mklinger GmbH and its suppliers and are protected
 * by trade secret or copyright law. Dissemination of this
 * information or reproduction of this material is strictly forbidden
 * unless prior written permission is obtained from mklinger GmbH.
 */
package de.mklinger.qetcher.liferay.client.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.portal.configuration.metatype.bnd.util.ConfigurableUtil;
import com.liferay.portal.kernel.util.MimeTypesUtil;

import de.mklinger.qetcher.client.model.v1.AvailableConversion;
import de.mklinger.qetcher.client.model.v1.Conversion;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.liferay.client.QetcherClientService;
import de.mklinger.qetcher.liferay.client.QetcherConversionsService;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
@Component(configurationPid = QetcherConfiguration.ID, configurationPolicy = ConfigurationPolicy.REQUIRE)
public class QetcherConversionsServiceImpl implements QetcherConversionsService {
	private static final String[] EMPTY_STRING_ARRAY = new String[0];
	private static final Logger LOG = LoggerFactory.getLogger(QetcherConversionsServiceImpl.class);

	private QetcherClientService clientService;
	private QetcherConversionsCache conversionsCache;
	private QetcherConfiguration configuration;

	@Reference
	public void setClientService(QetcherClientService clientService) {
		this.clientService = clientService;
	}

	@Reference
	public void setConversionsCache(QetcherConversionsCache conversionsCache) {
		this.conversionsCache = conversionsCache;
	}

	@Activate
	@Modified
	public synchronized void activate(final Map<String, Object> configurationProperties) {
		LOG.info("Configuration was set");
		this.configuration = ConfigurableUtil.createConfigurable(QetcherConfiguration.class, configurationProperties);
	}

	@Override
	public String[] getTargetExtensionsForExtension(final String extension) {
		if (extension == null || extension.isEmpty()) {
			return EMPTY_STRING_ARRAY;
		}

		String[] result = conversionsCache.getTargetExtensionsFromCache(extension);
		if (result == null) {
			synchronized (this) {
				// double check in synchronized block
				result = conversionsCache.getTargetExtensionsFromCache(extension);
				if (result == null) {
					result = loadTargetExtensions(extension);
					conversionsCache.putTargetExtensionsToCache(extension, result);
				}
			}
		}
		return result;
	}

	private String[] loadTargetExtensions(final String extension) {
		String[] newResult;
		final Map<String, Set<String>> supportedConversionsExtensions = getSupportedConversionsExtensions();
		Set<String> targetExtensions = supportedConversionsExtensions.get(extension);
		if (targetExtensions == null || targetExtensions.isEmpty()) {
			LOG.info("Found no conversions for source extension {}", extension);
			newResult = EMPTY_STRING_ARRAY;
		} else {
			LOG.info("Found {} conversions for source extension {}", targetExtensions.size(), extension);
			final Set<String> extensionWhitelist = getExtensionWhitelist(extension);
			if (extensionWhitelist != null && !extensionWhitelist.isEmpty() || targetExtensions.contains(extension)) {
				targetExtensions = new HashSet<>(targetExtensions);
				targetExtensions.remove(extension);
				if (extensionWhitelist != null) {
					targetExtensions.retainAll(extensionWhitelist);
				}
			}
			if (targetExtensions.isEmpty()) {
				newResult = EMPTY_STRING_ARRAY;
			} else {
				newResult = targetExtensions.toArray(new String[targetExtensions.size()]);
			}
		}
		return newResult;
	}

	/** Protected for unit test. */
	protected Set<String> getExtensionWhitelist(final String extension) {
		final Set<String> extensionWhitelist;
		if (configuration.conversionExtensionsSinglepage().contains(extension)) {
			extensionWhitelist = configuration.conversionExtensionsWhitelistSinglepage();
		} else {
			extensionWhitelist = configuration.conversionExtensionsWhitelistMultipage();
		}
		return extensionWhitelist;
	}

	private Map<String, Set<String>> getSupportedConversionsExtensions() {
		final Map<String, Set<String>> supportedConversionsExtensions = new HashMap<>();
		final List<AvailableConversion> availableConversions = getAvailableConversions();
		for (final AvailableConversion availableConversion : availableConversions) {
			final Conversion conversion = availableConversion.getConversion();
			final MediaType from = conversion.getFrom();
			final Set<String> fromExtensions = getExtensionsWithoutLeadingDot(from);
			if (fromExtensions != null) {
				for (final String fromExtension : fromExtensions) {
					addExtensions(supportedConversionsExtensions, fromExtension, conversion.getTo());
				}
			}
		}
		return supportedConversionsExtensions;
	}

	private void addExtensions(final Map<String, Set<String>> supportedConversionsExtensions, final String fromExtension, final MediaType to) {
		final Set<String> toExtensions = supportedConversionsExtensions.computeIfAbsent(fromExtension,
				unused -> new HashSet<>());

		final Set<String> newToExtensions = getExtensionsWithoutLeadingDot(to);
		if (newToExtensions != null) {
			toExtensions.addAll(newToExtensions);
		}
	}

	private Set<String> getExtensionsWithoutLeadingDot(final MediaType mediaType) {
		final Set<String> extensions = getExtensionsForMediaType(mediaType);
		if (extensions == null || extensions.isEmpty()) {
			return extensions;
		} else {
			final LinkedHashSet<String> extensionsWithoutLeadingDot = new LinkedHashSet<>(extensions.size());
			for (final String extension : extensions) {
				if (extension != null) {
					extensionsWithoutLeadingDot.add(getExtensionWithoutLeadingDot(extension));
				}
			}
			return extensionsWithoutLeadingDot;
		}
	}

	private Set<String> getExtensionsForMediaType(final MediaType from) {
		Set<String> extensions = conversionsCache.getExtensionsForMediaTypeFromCache(from);
		if (extensions == null) {
			synchronized (this) {
				// double check in synchronized block
				extensions = conversionsCache.getExtensionsForMediaTypeFromCache(from);
				if (extensions == null) {
					extensions = loadExtensionsForMediaType(from);
					conversionsCache.putExtensionsForMediaTypeToCache(from, extensions);
				}
			}
		}
		return extensions;
	}

	private Set<String> loadExtensionsForMediaType(MediaType from) {
		return MimeTypesUtil.getExtensions(from.getFullType());
	}

	private String getExtensionWithoutLeadingDot(final String extension) {
		if (extension.startsWith(".") && extension.length() > 1) {
			return extension.substring(1);
		} else {
			return extension;
		}
	}

	@Override
	public List<AvailableConversion> getAvailableConversions() {
		List<AvailableConversion> conversions = conversionsCache.getAvailableConversionsFromCache();
		if (conversions == null) {
			synchronized (this) {
				// double check in synchronized block
				conversions = conversionsCache.getAvailableConversionsFromCache();
				if (conversions == null) {
					conversions = loadAvailableConversions();
					if (conversions == null || conversions.isEmpty()) {
						LOG.warn("No conversions available");
						return Collections.emptyList();
					}
					conversionsCache.putAvailableConversionsToCache(conversions);
				}
			}
		}
		return conversions;
	}

	/** Protected for unit test. */
	protected List<AvailableConversion> loadAvailableConversions() {
		return clientService.client().getAvailableConversions().join();
	}

	/**
	 * @param registry The first part of the media type, e.g. "audio".
	 */
	@Override
	public Set<String> getSourceMediaTypesForRegistry(final String registry) {
		HashSet<String> result = conversionsCache.getSupportedSourceMimeTypesFromCache(registry);
		if (result == null) {
			synchronized (this) {
				// double check in synchronized block
				result = conversionsCache.getSupportedSourceMimeTypesFromCache(registry);
				if (result == null) {
					result = loadSupportedSourceMimeTypes(registry);
					conversionsCache.putSupportedSourceMimeTypesToCache(registry, result);
				}
			}
		}
		return Collections.unmodifiableSet(result);
	}

	private HashSet<String> loadSupportedSourceMimeTypes(final String registry) {
		final MediaType registryMediaType = new MediaType(registry, "*");
		final List<AvailableConversion> availableConversions = getAvailableConversions();
		final HashSet<String> mimeTypes = new HashSet<>();
		for (final AvailableConversion availableConversion : availableConversions) {
			final MediaType from = availableConversion.getConversion().getFrom();
			if (!from.isWildcardType() && !from.isWildcardSubtype() && from.isCompatible(registryMediaType)) {
				mimeTypes.add(from.getFullType());
			}
		}
		return mimeTypes;
	}
}
