package de.mklinger.qetcher.liferay.client.impl;

import java.io.Serializable;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.mklinger.qetcher.client.model.v1.AvailableConversion;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.liferay.abstraction.CacheTool;

@Component(service = QetcherConversionsCache.class)
public class QetcherConversionsCache {
	private static final String CACHE_NAME = "de.mklinger.qetcher.liferay.ConversionsCache"; // keep this name
	private static final String TARGET_EXTENSIONS_PREFIX = "targetExtensions_";
	private static final String EXTENSIONS_FOR_MEDIA_TYPE_PREFIX = "extensionsForMediaType_";
	private static final String CONVERSIONS_KEY = "conversions";
	private static final String SOURCE_MEDIA_TYPES_REGISTRY_PREFIX = "sourceMediaTypesRegistry_";

	private static final Logger LOG = LoggerFactory.getLogger(QetcherConversionsCache.class);

	private CacheTool cacheTool;

	@Reference
	public void setCacheTool(CacheTool cacheTool) {
		this.cacheTool = cacheTool;
	}

	public String[] getTargetExtensionsFromCache(final String extension) {
		final String cacheKey = TARGET_EXTENSIONS_PREFIX + extension;
		final String[] cachedResult = (String[])cacheTool.get(CACHE_NAME, cacheKey);
		if (LOG.isDebugEnabled()) {
			if (cachedResult != null) {
				LOG.debug("Cache hit for {}", cacheKey);
			} else {
				LOG.debug("Cache miss for {}", cacheKey);
			}
		}
		return cachedResult;
	}

	public void putTargetExtensionsToCache(final String extension, final String[] targetExtensions) {
		final String cacheKey = TARGET_EXTENSIONS_PREFIX + extension;
		cacheTool.put(CACHE_NAME, cacheKey, targetExtensions);
	}

	public Set<String> getExtensionsForMediaTypeFromCache(MediaType from) {
		final String cacheKey = EXTENSIONS_FOR_MEDIA_TYPE_PREFIX + from.getFullType();

		@SuppressWarnings("unchecked")
		final Set<String> extensions = (Set<String>)cacheTool.get(CACHE_NAME, cacheKey);

		if (LOG.isDebugEnabled()) {
			if (extensions == null) {
				LOG.debug("Cache miss for {}", cacheKey);
			} else {
				LOG.debug("Cache hit for {}", cacheKey);
			}
		}

		return extensions;
	}

	public void putExtensionsForMediaTypeToCache(MediaType from, Set<String> extensions) {
		final String cacheKey = EXTENSIONS_FOR_MEDIA_TYPE_PREFIX + from.getFullType();

		cacheTool.put(CACHE_NAME, cacheKey, (Serializable) extensions);
	}

	public List<AvailableConversion> getAvailableConversionsFromCache() {
		@SuppressWarnings("unchecked")
		final List<AvailableConversion> conversions = (List<AvailableConversion>)cacheTool.get(CACHE_NAME, CONVERSIONS_KEY);
		if (LOG.isDebugEnabled()) {
			if (conversions == null) {
				LOG.debug("Cache miss for {}", CONVERSIONS_KEY);
			} else {
				LOG.debug("Cache hit for {}", CONVERSIONS_KEY);
			}
		}
		return conversions;
	}

	public void putAvailableConversionsToCache(final List<AvailableConversion> conversions) {
		cacheTool.put(CACHE_NAME, CONVERSIONS_KEY, (Serializable) conversions);
	}

	public HashSet<String> getSupportedSourceMimeTypesFromCache(final String registry) {
		final String cacheKey = SOURCE_MEDIA_TYPES_REGISTRY_PREFIX + registry;
		@SuppressWarnings("unchecked")
		final HashSet<String> cachedResult = (HashSet<String>)cacheTool.get(CACHE_NAME, cacheKey);
		if (LOG.isDebugEnabled()) {
			if (cachedResult != null) {
				LOG.debug("Cache hit for {}", cacheKey);
			} else {
				LOG.debug("Cache miss for {}", cacheKey);
			}
		}
		return cachedResult;
	}

	public void putSupportedSourceMimeTypesToCache(final String registry, final HashSet<String> mimeTypes) {
		final String cacheKey = SOURCE_MEDIA_TYPES_REGISTRY_PREFIX + registry;
		cacheTool.put(CACHE_NAME, cacheKey, mimeTypes);
	}
}
