/*
 * Copyright 2013-present mklinger GmbH - http://www.mklinger.de
 *
 * All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of mklinger GmbH and its suppliers, if any.
 * The intellectual and technical concepts contained herein are
 * proprietary to mklinger GmbH and its suppliers and are protected
 * by trade secret or copyright law. Dissemination of this
 * information or reproduction of this material is strictly forbidden
 * unless prior written permission is obtained from mklinger GmbH.
 */
package de.mklinger.qetcher.liferay.client.impl;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.portal.kernel.util.MimeTypesUtil;

import de.mklinger.qetcher.client.model.v1.AvailableConversion;
import de.mklinger.qetcher.client.model.v1.Conversion;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.liferay.abstraction.CacheTool;
import de.mklinger.qetcher.liferay.client.impl.abstraction.LiferayAbstractionFactorySupplier;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class Conversions {
	private static final Conversions INSTANCE = new Conversions();
	private static final String CACHE_NAME = "de.mklinger.qetcher.liferay.ConversionsCache"; // keep this name
	private static final String CONVERSIONS_EXTENSIONS_PREFIX = "conversionsExtensions_";
	private static final String CONVERSIONS_KEY = "conversions";
	private static final String SOURCE_MEDIA_TYPES_REGISTRY_PREFIX = "sourceMediaTypesRegistry_";
	private static final String[] EMPTY_STRING_ARRAY = new String[0];
	private static final Logger LOG = LoggerFactory.getLogger(Conversions.class);

	/** Protected for unit test. */
	protected Conversions() {
	}

	public static Conversions getInstance() {
		return INSTANCE;
	}

	public String[] getTargetExtensionsForExtension(final String extension) {
		if (extension == null || extension.isEmpty()) {
			return EMPTY_STRING_ARRAY;
		}

		String[] result = getTargetExtensionsFromCache(extension);
		if (result == null) {
			synchronized (this) {
				// double check in synchronized block
				result = getTargetExtensionsFromCache(extension);
				if (result == null) {
					result = loadTargetExtensions(extension);
					putTargetExtensionsToCache(extension, result);
				}
			}
		}
		return result;
	}

	private String[] loadTargetExtensions(final String extension) {
		String[] newResult;
		final Map<String, Set<String>> supportedConversionsExtensions = getSupportedConversionsExtensions();
		Set<String> targetExtensions = supportedConversionsExtensions.get(extension);
		if (targetExtensions == null || targetExtensions.isEmpty()) {
			LOG.info("Found no conversions for source extension {}", extension);
			newResult = EMPTY_STRING_ARRAY;
		} else {
			LOG.info("Found {} conversions for source extension {}", targetExtensions.size(), extension);
			final Set<String> extensionWhitelist = getExtensionWhitelist(extension);
			if (extensionWhitelist != null && !extensionWhitelist.isEmpty() || targetExtensions.contains(extension)) {
				targetExtensions = new HashSet<>(targetExtensions);
				targetExtensions.remove(extension);
				if (extensionWhitelist != null) {
					targetExtensions.retainAll(extensionWhitelist);
				}
			}
			if (targetExtensions.isEmpty()) {
				newResult = EMPTY_STRING_ARRAY;
			} else {
				newResult = targetExtensions.toArray(new String[targetExtensions.size()]);
			}
		}
		return newResult;
	}

	/** Protected for unit test. */
	protected Set<String> getExtensionWhitelist(final String extension) {
		final Set<String> extensionWhitelist;
		final LiferayClientConfiguration configuration = LiferayClientConfiguration.getInstance();
		if (configuration.getConversionExtensionsSinglepage().contains(extension)) {
			extensionWhitelist = configuration.getConversionExtensionsWhitelistSinglepage();
		} else {
			extensionWhitelist = configuration.getConversionExtensionsWhitelistMultipage();
		}
		return extensionWhitelist;
	}

	private String[] getTargetExtensionsFromCache(final String extension) {
		final String cacheKey = CONVERSIONS_EXTENSIONS_PREFIX + extension;
		final String[] cachedResult = (String[])getCacheTool().get(CACHE_NAME, cacheKey);
		if (LOG.isDebugEnabled()) {
			if (cachedResult != null) {
				LOG.debug("Cache hit for {}", cacheKey);
			} else {
				LOG.debug("Cache miss for {}", cacheKey);
			}
		}
		return cachedResult;
	}

	private void putTargetExtensionsToCache(final String extension, final String[] targetExtensions) {
		final String cacheKey = CONVERSIONS_EXTENSIONS_PREFIX + extension;
		getCacheTool().put(CACHE_NAME, cacheKey, targetExtensions);
	}

	private Map<String, Set<String>> getSupportedConversionsExtensions() {
		final Map<String, Set<String>> supportedConversionsExtensions = new HashMap<>();
		final List<AvailableConversion> availableConversions = getAvailableConversions();
		for (final AvailableConversion availableConversion : availableConversions) {
			final Conversion conversion = availableConversion.getConversion();
			final MediaType from = conversion.getFrom();
			final Set<String> fromExtensions = getExtensionsWithoutLeadingDot(from);
			if (fromExtensions != null) {
				for (final String fromExtension : fromExtensions) {
					addExtensions(supportedConversionsExtensions, fromExtension, conversion.getTo());
				}
			}
		}
		return supportedConversionsExtensions;
	}

	private void addExtensions(final Map<String, Set<String>> supportedConversionsExtensions, final String fromExtension, final MediaType to) {
		final Set<String> toExtensions = supportedConversionsExtensions.computeIfAbsent(fromExtension,
				unused -> new HashSet<>());

		final Set<String> newToExtensions = getExtensionsWithoutLeadingDot(to);
		if (newToExtensions != null) {
			toExtensions.addAll(newToExtensions);
		}
	}

	private Set<String> getExtensionsWithoutLeadingDot(final MediaType from) {
		final Set<String> extensions = MimeTypesUtil.getExtensions(from.getFullType());
		if (extensions == null || extensions.isEmpty()) {
			return extensions;
		} else {
			final LinkedHashSet<String> extensionsWithoutLeadingDot = new LinkedHashSet<>(extensions.size());
			for (final String extension : extensions) {
				if (extension != null) {
					extensionsWithoutLeadingDot.add(getExtensionWithoutLeadingDot(extension));
				}
			}
			return extensionsWithoutLeadingDot;
		}
	}

	private String getExtensionWithoutLeadingDot(final String extension) {
		if (extension.startsWith(".") && extension.length() > 1) {
			return extension.substring(1);
		} else {
			return extension;
		}
	}

	public List<AvailableConversion> getAvailableConversions() {
		List<AvailableConversion> conversions = getAvailableConversionsFromCache();
		if (conversions == null) {
			synchronized (this) {
				// double check in synchronized block
				conversions = getAvailableConversionsFromCache();
				if (conversions == null) {
					conversions = loadAvailableConversions();
					if (conversions == null || conversions.isEmpty()) {
						LOG.warn("No conversions available");
						return Collections.emptyList();
					}
					putAvailableConversionsToCache(conversions);
				}
			}
		}
		return conversions;
	}

	private List<AvailableConversion> getAvailableConversionsFromCache() {
		@SuppressWarnings("unchecked")
		final List<AvailableConversion> conversions = (List<AvailableConversion>)getCacheTool().get(CACHE_NAME, CONVERSIONS_KEY);
		if (LOG.isDebugEnabled()) {
			if (conversions == null) {
				LOG.debug("Cache miss for {}", CONVERSIONS_KEY);
			} else {
				LOG.debug("Cache hit for {}", CONVERSIONS_KEY);
			}
		}
		return conversions;
	}

	private void putAvailableConversionsToCache(final List<AvailableConversion> conversions) {
		getCacheTool().put(CACHE_NAME, CONVERSIONS_KEY, (Serializable) conversions);
	}

	/** Protected for unit test. */
	protected List<AvailableConversion> loadAvailableConversions() {
		return QetcherLiferayServiceUtil.getAvailableConversions();
	}

	/**
	 * @param registry The first part of the media type, e.g. "audio".
	 */
	public Set<String> getSourceMediaTypesForRegistry(final String registry) {
		HashSet<String> result = getSupportedSourceMimeTypesFromCache(registry);
		if (result == null) {
			synchronized (this) {
				// double check in synchronized block
				result = getSupportedSourceMimeTypesFromCache(registry);
				if (result == null) {
					result = loadSupportedSourceMimeTypes(registry);
					putSupportedSourceMimeTypesToCache(registry, result);
				}
			}
		}
		return Collections.unmodifiableSet(result);
	}

	private HashSet<String> getSupportedSourceMimeTypesFromCache(final String registry) {
		final String cacheKey = SOURCE_MEDIA_TYPES_REGISTRY_PREFIX + registry;
		@SuppressWarnings("unchecked")
		final HashSet<String> cachedResult = (HashSet<String>)getCacheTool().get(CACHE_NAME, cacheKey);
		if (LOG.isDebugEnabled()) {
			if (cachedResult != null) {
				LOG.debug("Cache hit for {}", cacheKey);
			} else {
				LOG.debug("Cache miss for {}", cacheKey);
			}
		}
		return cachedResult;
	}

	private void putSupportedSourceMimeTypesToCache(final String registry, final HashSet<String> mimeTypes) {
		final String cacheKey = SOURCE_MEDIA_TYPES_REGISTRY_PREFIX + registry;
		getCacheTool().put(CACHE_NAME, cacheKey, mimeTypes);
	}

	/** Protected for unit test. */
	protected CacheTool getCacheTool() {
		return LiferayAbstractionFactorySupplier.getInstance().getCacheTool();
	}

	private HashSet<String> loadSupportedSourceMimeTypes(final String registry) {
		final MediaType registryMediaType = new MediaType(registry, "*");
		final List<AvailableConversion> availableConversions = getAvailableConversions();
		final HashSet<String> mimeTypes = new HashSet<>();
		for (final AvailableConversion availableConversion : availableConversions) {
			final MediaType from = availableConversion.getConversion().getFrom();
			if (!from.isWildcardType() && !from.isWildcardSubtype() && from.isCompatible(registryMediaType)) {
				mimeTypes.add(from.getFullType());
			}
		}
		return mimeTypes;
	}
}
