/*
 * Copyright 2013-present mklinger GmbH - http://www.mklinger.de
 *
 * All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of mklinger GmbH and its suppliers, if any.
 * The intellectual and technical concepts contained herein are
 * proprietary to mklinger GmbH and its suppliers and are protected
 * by trade secret or copyright law. Dissemination of this
 * information or reproduction of this material is strictly forbidden
 * unless prior written permission is obtained from mklinger GmbH.
 */
package de.mklinger.qetcher.liferay.client.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.portal.kernel.lar.PortletDataContext;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.messaging.DestinationNames;
import com.liferay.portal.kernel.repository.model.FileEntry;
import com.liferay.portal.kernel.repository.model.FileVersion;
import com.liferay.portal.kernel.util.FileUtil;
import com.liferay.portal.kernel.util.StreamUtil;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.kernel.xml.Element;
import com.liferay.portal.repository.liferayrepository.model.LiferayFileVersion;
import com.liferay.portal.util.PropsValues;
import com.liferay.portlet.documentlibrary.NoSuchFileEntryException;
import com.liferay.portlet.documentlibrary.util.AudioProcessor;
import com.liferay.portlet.documentlibrary.util.DLUtil;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class AudioProcessorImpl extends MediaRegistryProcessor implements AudioProcessor {
	private static final Logger LOG = LoggerFactory.getLogger(AudioProcessorImpl.class);

	public AudioProcessorImpl() {
		super("audio");
	}

	private void _generateAudioQetcher(final FileVersion fileVersion, final File srcFile, final File destFile, final String containerType) throws Exception {
		if (hasPreview(fileVersion, containerType)) {
			return;
		}

		final String sourceExtension = getExtension(fileVersion);
		if (sourceExtension == null) {
			return;
		}

		try (FileInputStream in = new FileInputStream(srcFile)) {
			try (FileOutputStream out = new FileOutputStream(destFile)) {
				if ("mp3".equals(sourceExtension) && sourceExtension.equals(containerType)) {
					// only reuse existing file for mp3, other containers could contain other codecs.
					LOG.info("Using original file as preview for type {}", sourceExtension);
					IOUtils.copy(in, out);
				} else {
					LOG.debug("Generating audio preview from {} to {}", sourceExtension, containerType);
					QetcherLiferayServiceUtil.convert(in, out, sourceExtension, containerType, null, getReferer(fileVersion));
					LOG.debug("Generating audio preview done.");
				}
			}
		}

		addFileToStore(
				fileVersion.getCompanyId(), PREVIEW_PATH,
				getPreviewFilePath(fileVersion, containerType), destFile);
	}

	private String getReferer(final FileVersion fileVersion) {
		return "companyId=" + fileVersion.getCompanyId() + ",fileEntryId=" + fileVersion.getFileEntryId();
	}

	private String getExtension(final FileVersion fileVersion) {
		final String mimeTypeString = fileVersion.getMimeType();
		LOG.debug("Original source file mime type: {}", mimeTypeString);
		LOG.debug("Original source file extension: {}", fileVersion.getExtension());
		String sourceExtension = MimeTypesImpl.getInstance().getExtensionContentType(mimeTypeString);
		if (sourceExtension == null) {
			sourceExtension = fileVersion.getExtension();
		}
		if (sourceExtension == null || sourceExtension.isEmpty()) {
			LOG.warn("Could not determine source file extension for file entry {}", fileVersion.getFileEntryId());
			return null;
		}
		return sourceExtension;
	}

	// -----------

	@Override
	public void afterPropertiesSet() {
		FileUtil.mkdirs(PREVIEW_TMP_PATH);
	}

	@Override
	public void generateAudio(final FileVersion sourceFileVersion, final FileVersion destinationFileVersion) throws Exception {
		_generateAudio(sourceFileVersion, destinationFileVersion);
	}

	@Override
	public Set<String> getAudioMimeTypes() {
		return getSupportedSourceMimeTypes();
	}

	@Override
	public InputStream getPreviewAsStream(final FileVersion fileVersion, final String type) throws Exception {
		return doGetPreviewAsStream(fileVersion, type);
	}

	@Override
	public long getPreviewFileSize(final FileVersion fileVersion, final String type) throws Exception {
		return doGetPreviewFileSize(fileVersion, type);
	}

	@Override
	public boolean hasAudio(final FileVersion fileVersion) {
		boolean hasAudio = false;

		try {
			hasAudio = _hasAudio(fileVersion);

			if (!hasAudio && isSupported(fileVersion)) {
				_queueGeneration(null, fileVersion);
			}
		} catch (final Exception e) {
			_log.error(e, e);
		}

		return hasAudio;
	}

	@Override
	public boolean isAudioSupported(final FileVersion fileVersion) {
		return isSupported(fileVersion);
	}

	@Override
	public boolean isAudioSupported(final String mimeType) {
		return isSupported(mimeType);
	}

	@Override
	public boolean isSupported(final String mimeType) {
		if (Validator.isNull(mimeType)) {
			return false;
		}
		return getAudioMimeTypes().contains(mimeType);
	}

	@Override
	public void trigger(final FileVersion sourceFileVersion, final FileVersion destinationFileVersion) {
		super.trigger(sourceFileVersion, destinationFileVersion);
		_queueGeneration(sourceFileVersion, destinationFileVersion);
	}

	@Override
	protected void doExportGeneratedFiles(final PortletDataContext portletDataContext, final FileEntry fileEntry, final Element fileEntryElement) throws Exception {
		exportPreviews(portletDataContext, fileEntry, fileEntryElement);
	}

	@Override
	protected void doImportGeneratedFiles(
			final PortletDataContext portletDataContext, final FileEntry fileEntry,
			final FileEntry importedFileEntry, final Element fileEntryElement)
					throws Exception {

		importPreviews(portletDataContext, fileEntry, importedFileEntry, fileEntryElement);
	}

	protected void exportPreviews(
			final PortletDataContext portletDataContext, final FileEntry fileEntry,
			final Element fileEntryElement)
					throws Exception {

		final FileVersion fileVersion = fileEntry.getFileVersion();

		if (!isSupported(fileVersion) || !hasPreviews(fileVersion)) {
			return;
		}

		if (!portletDataContext.isPerformDirectBinaryImport()) {
			for (final String previewType : _PREVIEW_TYPES) {
				exportPreview(
						portletDataContext, fileEntry, fileEntryElement,
						"audio", previewType);
			}
		}
	}

	@Override
	protected List<Long> getFileVersionIds() {
		return _fileVersionIds;
	}

	@Override
	protected String getPreviewType(final FileVersion fileVersion) {
		return _PREVIEW_TYPES[0];
	}

	@Override
	protected String[] getPreviewTypes() {
		return _PREVIEW_TYPES;
	}

	@Override
	protected String getThumbnailType(final FileVersion fileVersion) {
		return null;
	}

	protected void importPreviews(
			final PortletDataContext portletDataContext, final FileEntry fileEntry,
			final FileEntry importedFileEntry, final Element fileEntryElement)
					throws Exception {

		for (final String previewType : _PREVIEW_TYPES) {
			importPreview(
					portletDataContext, fileEntry, importedFileEntry,
					fileEntryElement, "audio", previewType);
		}
	}

	private void _generateAudio(final FileVersion sourceFileVersion, final FileVersion destinationFileVersion) throws Exception {

		final String tempFileId = DLUtil.getTempFileId(
				destinationFileVersion.getFileEntryId(),
				destinationFileVersion.getVersion());

		final File[] previewTempFiles = new File[_PREVIEW_TYPES.length];

		for (int i = 0; i < _PREVIEW_TYPES.length; i++) {
			previewTempFiles[i] = getPreviewTempFile(
					tempFileId, _PREVIEW_TYPES[i]);
		}

		File audioTempFile = null;

		InputStream inputStream = null;

		try {
			if (sourceFileVersion != null) {
				copy(sourceFileVersion, destinationFileVersion);

				return;
			}

			audioTempFile = FileUtil.createTempFile(destinationFileVersion.getExtension());

			//			if (_hasAudio(fileVersion)) {
			//				return;
			//			}

			if (!hasPreviews(destinationFileVersion)) {
				File file = null;

				if (destinationFileVersion instanceof LiferayFileVersion) {
					try {
						final LiferayFileVersion liferayFileVersion =
								(LiferayFileVersion)destinationFileVersion;

						file = liferayFileVersion.getFile(false);
					}
					catch (final UnsupportedOperationException uoe) {
					}
				}

				if (file == null) {
					inputStream = destinationFileVersion.getContentStream(false);

					FileUtil.write(audioTempFile, inputStream);

					file = audioTempFile;
				}

				try {
					_generateAudio(destinationFileVersion, file, previewTempFiles);
				}
				catch (final Exception e) {
					_log.error(e, e);
				}
			}
		}
		catch (final NoSuchFileEntryException nsfee) {
		}
		finally {
			StreamUtil.cleanUp(inputStream);

			_fileVersionIds.remove(destinationFileVersion.getFileVersionId());

			for (final File previewTempFile : previewTempFiles) {
				FileUtil.delete(previewTempFile);
			}

			FileUtil.delete(audioTempFile);
		}
	}

	private void _generateAudio(final FileVersion fileVersion, final File srcFile, final File[] destFiles) {
		try {
			for (int i = 0; i < destFiles.length; i++) {
				_generateAudioQetcher(fileVersion, srcFile, destFiles[i], _PREVIEW_TYPES[i]);
			}
		}
		catch (final Exception e) {
			_log.error(e, e);
		}
	}

	private boolean _hasAudio(final FileVersion fileVersion) throws Exception {
		if (!isSupported(fileVersion)) {
			return false;
		}

		return hasPreviews(fileVersion);
	}

	private void _queueGeneration(final FileVersion sourceFileVersion, final FileVersion destinationFileVersion) {
		if (_fileVersionIds.contains(destinationFileVersion.getFileVersionId()) ||
				!isSupported(destinationFileVersion)) {

			return;
		}

		_fileVersionIds.add(destinationFileVersion.getFileVersionId());

		sendGenerationMessage(
				DestinationNames.DOCUMENT_LIBRARY_AUDIO_PROCESSOR,
				sourceFileVersion, destinationFileVersion);
	}

	private static final String[] _PREVIEW_TYPES = PropsValues.DL_FILE_ENTRY_PREVIEW_AUDIO_CONTAINERS;
	private static Log _log = LogFactoryUtil.getLog(AudioProcessorImpl.class);
	private final List<Long> _fileVersionIds = new Vector<>();
}