package de.mklinger.qetcher.liferay.client.impl.mediatype;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Properties;
import java.util.StringTokenizer;

import org.jsoup.UncheckedIOException;

import de.mklinger.qetcher.client.lists.Lists;
import de.mklinger.qetcher.client.maps.Maps;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class MediaTypesService {
	private static final MediaTypesService INSTANCE = new MediaTypesService();

	private volatile Map<String, List<String>> extensionsByMediaType = null;
	private volatile Properties mediaTypeByExtension = null;

	public static MediaTypesService getInstance() {
		return INSTANCE;
	}

	public Optional<String> getFileExtensionByMimeType(final String mediaType) {
		final List<String> extensions = getExtensionsByMediaType().get(mediaType);
		if (extensions == null || extensions.isEmpty()) {
			return Optional.empty();
		} else {
			return Optional.of(extensions.get(0));
		}
	}

	public List<String> getFileExtensionsByMimeType(final String mediaType) {
		return getExtensionsByMediaType().get(mediaType);
	}

	public Optional<String> getMimeTypeStringByExtension(final String extension) {
		return Optional.ofNullable(getMediaTypeByExtension()
				.getProperty(extension.toLowerCase()));
	}


	private Map<String, List<String>> getExtensionsByMediaType() {
		if (extensionsByMediaType == null) {
			synchronized (this) {
				if (extensionsByMediaType == null) {
					extensionsByMediaType = loadExtensionsByMediaType();
				}
			}
		}
		return extensionsByMediaType;
	}

	private Map<String, List<String>> loadExtensionsByMediaType() {
		try (final InputStream in = getClass().getResourceAsStream("extensions-by-mediatype.properties")) {
			final Properties p = new Properties();
			p.load(in);

			final Map<String, List<String>> map = Maps.newHashMap(p.size());

			for (final Entry<Object, Object> entry : p.entrySet()) {
				final String key = (String) entry.getKey();
				final String values = (String) entry.getValue();
				final List<String> extensions = new ArrayList<>();
				final StringTokenizer st = new StringTokenizer(values, ",");
				while (st.hasMoreTokens()) {
					final String extension = st.nextToken();
					extensions.add(extension);
				}
				map.put(key, Lists.newImmutableList(extensions));
			}

			return Maps.newImmutableMap(map);
		} catch (final IOException e) {
			throw new UncheckedIOException(e);
		}
	}

	private Properties getMediaTypeByExtension() {
		if (mediaTypeByExtension == null) {
			synchronized (this) {
				if (mediaTypeByExtension == null) {
					mediaTypeByExtension = loadMediaTypeByExtension();
				}
			}
		}
		return mediaTypeByExtension;
	}

	private Properties loadMediaTypeByExtension() {
		try (final InputStream in = getClass().getResourceAsStream("mediatype-by-extension.properties")) {
			final Properties p = new Properties();
			p.load(in);
			return p;

		} catch (final IOException e) {
			throw new UncheckedIOException(e);
		}
	}
}
