/*
 * Copyright 2013-present mklinger GmbH - http://www.mklinger.de
 *
 * All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of mklinger GmbH and its suppliers, if any.
 * The intellectual and technical concepts contained herein are
 * proprietary to mklinger GmbH and its suppliers and are protected
 * by trade secret or copyright law. Dissemination of this
 * information or reproduction of this material is strictly forbidden
 * unless prior written permission is obtained from mklinger GmbH.
 */
package de.mklinger.qetcher.liferay.client.impl;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.mklinger.qetcher.liferay.abstraction.FileVersion;
import de.mklinger.qetcher.liferay.abstraction.LiferayException;
import de.mklinger.qetcher.liferay.client.impl.abstraction.LiferayAbstractionFactory;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class PreviewOutputStream extends FileOutputStream {
	private static final Logger LOG = LoggerFactory.getLogger(PreviewOutputStream.class);

	private final boolean storeThumbnail;
	private final boolean storePreview;
	private final PreviewProcessorBridge processor;
	private final File tempFile;
	private final FileVersion fileVersion;
	private final int page;
	private boolean closed;

	public PreviewOutputStream(final FileVersion fileVersion, final int page, final boolean storeThumbnail, final boolean storePreview, final PreviewProcessorBridge processor) throws IOException {
		this(getTempFile(fileVersion, page, processor), page, storeThumbnail, storePreview, fileVersion, processor);
	}

	private PreviewOutputStream(final File file, final int page, final boolean storeThumbnail, final boolean storePreview, final FileVersion fileVersion, final PreviewProcessorBridge processor) throws FileNotFoundException {
		super(file);
		this.tempFile = file;
		this.page = page;
		this.fileVersion = fileVersion;
		this.storeThumbnail = storeThumbnail;
		this.storePreview = storePreview;
		this.processor = processor;
	}

	private static File getTempFile(final FileVersion fileVersion, final int page, final PreviewProcessorBridge processor) throws IOException {
		final String tempFileId = LiferayAbstractionFactory.getInstance().getDLTool().getTempFileId(fileVersion);
		final File tempFile = processor.getPreviewTempFile(tempFileId, page);
		tempFile.createNewFile();
		return tempFile;
	}

	@Override
	public void close() throws IOException {
		if (!closed) {
			closed = true;
			super.close();
			if (storeThumbnail) {
				try {
					LOG.debug("Storing thumbnail");
					processor.storeThumbnailImages(fileVersion, tempFile);
				} catch (final Exception e) {
					throw new RuntimeException("Error storing thumbnail image", e);
				}
			}
			if (storePreview) {
				try {
					LOG.debug("Storing preview page {}", page);
					final String previewPath = LiferayAbstractionFactory.getInstance().getDLTool().getPreviewPath();
					processor.addFileToStore(fileVersion.getCompanyId(), previewPath, processor.getPreviewFilePath(fileVersion, page), tempFile);
				} catch (final LiferayException e) {
					throw new RuntimeException("Error storing preview image", e);
				}
			}
		}
	}
}