/*
 * Copyright 2013-present mklinger GmbH - http://www.mklinger.de
 *
 * All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of mklinger GmbH and its suppliers, if any.
 * The intellectual and technical concepts contained herein are
 * proprietary to mklinger GmbH and its suppliers and are protected
 * by trade secret or copyright law. Dissemination of this
 * information or reproduction of this material is strictly forbidden
 * unless prior written permission is obtained from mklinger GmbH.
 */
package de.mklinger.qetcher.liferay.client.impl;

import static de.mklinger.qetcher.liferay.client.impl.QetcherPropsKeys.*;

import java.util.Collections;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import de.mklinger.micro.annotations.VisibleForTesting;
import de.mklinger.qetcher.client.sets.Sets;
import de.mklinger.qetcher.liferay.abstraction.PropsTool;
import de.mklinger.qetcher.liferay.client.impl.abstraction.LiferayAbstractionFactory;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class LiferayClientConfiguration {
	private static final String TMP_DIR = "java.io.tmpdir";

	private static volatile LiferayClientConfiguration instance;

	public static LiferayClientConfiguration getInstance() {
		if (instance == null) {
			synchronized (LiferayClientConfiguration.class) {
				if (instance == null) {
					instance = new LiferayClientConfiguration();
				}
			}
		}
		return instance;
	}

	private final String[] serviceAddresses;

	private final String keyStoreLocation;
	private final Optional<String> keyStorePassword;
	private final Optional<String> keyStoreType;
	private final Optional<String> keyPassword;

	private final String trustStoreLocation;
	private final Optional<String> trustStorePassword;
	private final Optional<String> trustStoreType;

	private final String documentConversionTargetPath;
	private final Set<String> conversionExtensionsSinglepage;
	private final Set<String> conversionExtensionsWhitelistSinglepage;
	private final Set<String> conversionExtensionsWhitelistMultipage;
	private final boolean storePdfAsPreview;
	private final boolean inlineCssForHtml;
	private final boolean inlineJsForHtml;
	private final boolean inlineExternalImagesForHtml;
	private final boolean inlineInternalImagesForHtml;
	private final long jobWaitTimeoutMillis;
	private final long jobCancelTimeoutMillis;
	private final long jobDeleteTimeoutMillis;

	@VisibleForTesting
	protected LiferayClientConfiguration() {
		final PropsTool propsTool = LiferayAbstractionFactory.getInstance().getPropsTool();

		serviceAddresses = propsTool.getArray(SERVICE_ADDRESSES);

		keyStoreLocation = Objects.requireNonNull(propsTool.getString(KEY_STORE_LOCATION, null), () -> "Missing required Qetcher property " + KEY_STORE_LOCATION);
		keyStorePassword = Optional.ofNullable(propsTool.getString(KEY_STORE_PASSWORD, null));
		keyStoreType = Optional.ofNullable(propsTool.getString(KEY_STORE_TYPE, null));
		keyPassword = Optional.ofNullable(propsTool.getString(KEY_PASSWORD, null));

		trustStoreLocation = Objects.requireNonNull(propsTool.getString(TRUST_STORE_LOCATION, null), () -> "Missing required Qetcher property " + TRUST_STORE_LOCATION);
		trustStorePassword = Optional.ofNullable(propsTool.getString(TRUST_STORE_PASSWORD, null));
		trustStoreType = Optional.ofNullable(propsTool.getString(TRUST_STORE_TYPE, null));

		documentConversionTargetPath = propsTool.getSystemProperty(TMP_DIR) + "/liferay/document_conversion";

		conversionExtensionsSinglepage = getSet(CONVERSION_EXTENSIONS_SINGLEPAGE);
		conversionExtensionsWhitelistSinglepage = getSet(CONVERSION_EXTENSIONS_WHITELIST_SINGLEPAGE);
		conversionExtensionsWhitelistMultipage = getSet(CONVERSION_EXTENSIONS_WHITELIST_MULTIPAGE);

		storePdfAsPreview = propsTool.getBoolean(STORE_PDF_AS_PREVIEW, false);

		inlineCssForHtml = propsTool.getBoolean(INLINE_CSS_FOR_HTML, true);
		inlineJsForHtml = propsTool.getBoolean(INLINE_JS_FOR_HTML, true);
		inlineExternalImagesForHtml = propsTool.getBoolean(INLINE_EXTERNAL_IMAGES_FOR_HTML, true);
		inlineInternalImagesForHtml = propsTool.getBoolean(INLINE_INTERNAL_IMAGES_FOR_HTML, true);

		jobWaitTimeoutMillis = propsTool.getLong(JOB_WAIT_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS.convert(1, TimeUnit.HOURS));
		jobCancelTimeoutMillis = propsTool.getLong(JOB_CANCEL_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS.convert(2, TimeUnit.HOURS));
		jobDeleteTimeoutMillis = propsTool.getLong(JOB_DELETE_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS.convert(24, TimeUnit.HOURS));
	}

	private Set<String> getSet(final String key) {
		final PropsTool propsTool = LiferayAbstractionFactory.getInstance().getPropsTool();
		final String[] arr = propsTool.getArray(key);
		if (arr == null || arr.length == 0) {
			return Collections.emptySet();
		}
		return Sets.newHashSet(arr);
	}

	public String[] getServiceAddresses() {
		return serviceAddresses;
	}

	public String getKeyStoreLocation() {
		return keyStoreLocation;
	}

	public Optional<String> getKeyStorePassword() {
		return keyStorePassword;
	}

	public Optional<String> getKeyStoreType() {
		return keyStoreType;
	}

	public Optional<String> getKeyPassword() {
		return keyPassword;
	}

	public String getTrustStoreLocation() {
		return trustStoreLocation;
	}

	public Optional<String> getTrustStorePassword() {
		return trustStorePassword;
	}

	public Optional<String> getTrustStoreType() {
		return trustStoreType;
	}

	public String getDocumentConversionTargetPath() {
		return documentConversionTargetPath;
	}

	public Set<String> getConversionExtensionsSinglepage() {
		return conversionExtensionsSinglepage;
	}

	public Set<String> getConversionExtensionsWhitelistMultipage() {
		return conversionExtensionsWhitelistMultipage;
	}

	public Set<String> getConversionExtensionsWhitelistSinglepage() {
		return conversionExtensionsWhitelistSinglepage;
	}

	public boolean isStorePdfAsPreview() {
		return storePdfAsPreview;
	}

	public boolean isInlineCssForHtml() {
		return inlineCssForHtml;
	}

	public boolean isInlineJsForHtml() {
		return inlineJsForHtml;
	}

	public boolean isInlineExternalImagesForHtml() {
		return inlineExternalImagesForHtml;
	}

	public boolean isInlineInternalImagesForHtml() {
		return inlineInternalImagesForHtml;
	}

	public long getJobCancelTimeoutMillis() {
		return jobCancelTimeoutMillis;
	}

	public long getJobDeleteTimeoutMillis() {
		return jobDeleteTimeoutMillis;
	}

	public long getJobWaitTimeoutMillis() {
		return jobWaitTimeoutMillis;
	}
}
