package com.liferay.portal.util;

import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.portal.kernel.util.MimeTypes;

import de.mklinger.qetcher.client.model.v1.FileExtensionInfos;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.client.model.v1.MediaTypeInfo;
import de.mklinger.qetcher.liferay.abstraction.CacheTool;
import de.mklinger.qetcher.liferay.client.impl.QetcherLiferayServiceUtil;
import de.mklinger.qetcher.liferay.client.impl.abstraction.LiferayAbstractionFactorySupplier;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class MimeTypesImpl implements MimeTypes {
	private static final String APPLICATION_OCTETSTREAM = "application/octet-stream";

	private static final String CACHE_NAME = "de.mklinger.qetcher.liferay.MimeTypesCache"; // keep this name
	private static final String FILEEXTENSION_INFOS_KEY = "mediaTypesByExtension";
	private static final String EXTENSIONS_BY_MEDIATYPE_KEY = "extensionsByMediaType";

	private static final String UNUSABLEFILENAME = "unusablefilename";

	private static final boolean PREFER_QETCHER_FILENAME_MEDIA_TYPE = true; // Tika produces audio/x-ms-wma for wmv

	private static final Logger LOG = LoggerFactory.getLogger(MimeTypesImpl.class);

	private MimeTypes liferayMimeTypes = new LiferayMimeTypesImpl();

	/** For unit tests. */
	protected void setLiferayMimeTypes(final MimeTypes liferayMimeTypes) {
		this.liferayMimeTypes = liferayMimeTypes;
	}

	@Override
	public String getContentType(final File file) {
		if (PREFER_QETCHER_FILENAME_MEDIA_TYPE) {
			final Optional<String> contentType = getQetcherFilenameContentType(file.getName());
			if (contentType.isPresent()) {
				return contentType.get();
			}
		}
		return getFileContentType(file).orElseGet(() -> getFilenameContentType(file.getName()));
	}

	@Override
	public String getContentType(final File file, final String fileName) {
		if (PREFER_QETCHER_FILENAME_MEDIA_TYPE && (file != null || fileName != null)) {
			String actualFileName = fileName;
			if (actualFileName == null) {
				actualFileName = file.getName();
			}
			final Optional<String> contentType = getQetcherFilenameContentType(actualFileName);
			if (contentType.isPresent()) {
				return contentType.get();
			}
		}
		return getFileContentType(file).orElseGet(() -> getFilenameContentType(fileName));
	}

	@Override
	public String getContentType(final InputStream inputStream, final String fileName) {
		if (PREFER_QETCHER_FILENAME_MEDIA_TYPE && fileName != null) {
			final Optional<String> contentType = getQetcherFilenameContentType(fileName);
			if (contentType.isPresent()) {
				return contentType.get();
			}
		}
		return getInputStreamContentType(inputStream).orElseGet(() -> getFilenameContentType(fileName));
	}

	@Override
	public String getContentType(final String fileName) {
		return getFilenameContentType(fileName);
	}

	@Override
	public boolean isWebImage(final String mimeType) {
		return liferayMimeTypes.isWebImage(mimeType);
	}

	/** Protected for unit test. */
	protected Optional<String> getFileContentType(final File file) {
		if (file == null || !file.exists()) {
			return Optional.empty();
		}
		final String contentType = liferayMimeTypes.getContentType(file, UNUSABLEFILENAME);
		if (contentType == null || contentType.isEmpty() || APPLICATION_OCTETSTREAM.equals(contentType)) {
			return Optional.empty();
		} else {
			LOG.info("Have content type by file contents for file name {}: {}", file.getName(), contentType);
			return Optional.of(contentType);
		}
	}

	/** Protected for unit test. */
	protected Optional<String> getInputStreamContentType(final InputStream in) {
		if (in == null) {
			return Optional.empty();
		}
		final String contentType = liferayMimeTypes.getContentType(in, UNUSABLEFILENAME);
		if (contentType == null || contentType.isEmpty() || APPLICATION_OCTETSTREAM.equals(contentType)) {
			return Optional.empty();
		} else {
			LOG.info("Have content type by input stream contents: {}", contentType);
			return Optional.of(contentType);
		}
	}

	@Override
	public String getExtensionContentType(final String extension) {
		final FileExtensionInfos fileExtensions = getFileExtensions();

		return fileExtensions.getMediaType(extension)
				.map(mediaType -> {
					LOG.debug("Using media type from Qetcher for extension '{}': '{}'", extension, mediaType);
					return mediaType;
				})
				.map(MediaType::toString)
				.orElseGet(() -> liferayMimeTypes.getExtensionContentType(extension));
	}

	private String getFilenameContentType(final String filename) {
		return getQetcherFilenameContentType(filename).orElseGet(() -> liferayMimeTypes.getContentType(filename));
	}

	private Optional<String> getQetcherFilenameContentType(final String filename) {
		final FileExtensionInfos fileExtensions = getFileExtensions();

		final List<String> extensions = getFilenameExtensions(filename);
		for (final String extension : extensions) {
			final Optional<MediaType> mediaType = fileExtensions.getMediaType(extension);
			if (mediaType.isPresent()) {
				LOG.debug("Using media type from Qetcher for filename '{}': '{}'", filename, mediaType.get());
				return Optional.of(mediaType.toString());
			}
		}

		return Optional.empty();
	}

	/** Protected for unit test. */
	protected List<String> getFilenameExtensions(final String filename) {
		final String simpleName = FilenameUtils.getName(filename);
		final List<String> extensions = new ArrayList<>(2);

		int startIdx = 0;
		while (simpleName.length() > startIdx) {
			final int idx = simpleName.indexOf('.', startIdx);
			if (idx != -1 && simpleName.length() > idx + 1) {
				final String extension = simpleName.substring(idx + 1);
				if (extension.length() > 0 && !extension.startsWith(".")) {
					extensions.add(extension);
				}
				startIdx = idx + 1;
			} else {
				break;
			}
		}

		return extensions;
	}

	@Override
	public Set<String> getExtensions(final String contentType) {
		final LinkedHashSet<String> extensions = getExtensionByMediaType().get(contentType.toLowerCase());
		if (extensions != null) {
			return extensions;
		}

		return liferayMimeTypes.getExtensions(contentType);
	}

	public FileExtensionInfos getFileExtensions() {
		FileExtensionInfos fileExtensions = getFileExtensionsFromCache();
		if (fileExtensions == null) {
			synchronized (this) {
				// double check in synchronized block
				fileExtensions = getFileExtensionsFromCache();
				if (fileExtensions == null) {
					fileExtensions = loadFileExtensions();
					putFileExtensionsToCache(fileExtensions);
				}
			}
		}
		return fileExtensions;
	}

	private FileExtensionInfos getFileExtensionsFromCache() {
		try {
			final Object[] serializable = (Object[])getCacheTool().get(CACHE_NAME, FILEEXTENSION_INFOS_KEY);
			if (serializable == null) {
				LOG.debug("Cache miss for {}", FILEEXTENSION_INFOS_KEY);
				return null;
			} else {
				LOG.debug("Cache hit for {}", FILEEXTENSION_INFOS_KEY);
				return (FileExtensionInfos) serializable[0];
			}
		} catch (final Exception e) {
			LOG.warn("Error accessing cache", e);
			return null;
		}
	}

	private FileExtensionInfos loadFileExtensions() {
		LOG.info("Loading file extensions");
		return QetcherLiferayServiceUtil.getFileExtensions();
	}

	private void putFileExtensionsToCache(final FileExtensionInfos o) {
		final Object[] serializable = { o };
		getCacheTool().put(CACHE_NAME, FILEEXTENSION_INFOS_KEY, serializable);
	}

	public Map<String, LinkedHashSet<String>> getExtensionByMediaType() {
		Map<String, LinkedHashSet<String>> fileExtensions = getExtensionByMediaTypeFromCache();
		if (fileExtensions == null) {
			synchronized (this) {
				// double check in synchronized block
				fileExtensions = getExtensionByMediaTypeFromCache();
				if (fileExtensions == null) {
					fileExtensions = loadExtensionByMediaType();
					putExtensionByMediaTypeToCache(fileExtensions);
				}
			}
		}
		return fileExtensions;
	}

	@SuppressWarnings("unchecked")
	private Map<String, LinkedHashSet<String>> getExtensionByMediaTypeFromCache() {
		try {
			final Object[] serializable = (Object[])getCacheTool().get(CACHE_NAME, EXTENSIONS_BY_MEDIATYPE_KEY);
			if (serializable == null) {
				LOG.debug("Cache miss for {}", EXTENSIONS_BY_MEDIATYPE_KEY);
				return null;
			} else {
				LOG.debug("Cache hit for {}", EXTENSIONS_BY_MEDIATYPE_KEY);
				return (Map<String, LinkedHashSet<String>>) serializable[0];
			}
		} catch (final Exception e) {
			LOG.warn("Error accessing cache", e);
			return null;
		}
	}

	private Map<String, LinkedHashSet<String>> loadExtensionByMediaType() {
		LOG.info("Loading file media types");
		final List<MediaTypeInfo> mediaTypes = QetcherLiferayServiceUtil.getMediaTypes();

		final Map<String, LinkedHashSet<String>> m = newHashMap(mediaTypes.size());

		for (final MediaTypeInfo mediaTypeInfo : mediaTypes) {
			final List<String> extensions = mediaTypeInfo.getExtensions();
			if (extensions != null && !extensions.isEmpty()) {
				final String mediaTypeString = mediaTypeInfo.getMediaType().toString().toLowerCase();
				m.computeIfAbsent(mediaTypeString, s -> new LinkedHashSet<String>())
				.addAll(extensions);
			}
		}

		for (final MediaTypeInfo mediaTypeInfo : mediaTypes) {
			final List<String> extensions = mediaTypeInfo.getExtensions();
			final List<MediaType> aliases = mediaTypeInfo.getAliases();
			if (extensions != null && !extensions.isEmpty() && aliases != null && !aliases.isEmpty()) {
				for (final MediaType alias : aliases) {
					final String mediaTypeString = alias.toString().toLowerCase();
					m.computeIfAbsent(mediaTypeString, s -> new LinkedHashSet<String>())
					.addAll(extensions);
				}
			}
		}

		return m;
	}

	/**
	 * Returns a HashMap with a capacity that is sufficient to keep the map
	 * from being resized as long as it grows no larger than expectedSize
	 * with the default load factor (0.75).
	 */
	private static <K, V> HashMap<K, V> newHashMap(final int expectedSize) {
		// See code in java.util.HashSet.HashSet(Collection<? extends E>)
		return new HashMap<>(Math.max((int) (expectedSize / .75f) + 1, 4));
	}

	private void putExtensionByMediaTypeToCache(final Map<String, LinkedHashSet<String>> o) {
		final Object[] serializable = { o };
		getCacheTool().put(CACHE_NAME, EXTENSIONS_BY_MEDIATYPE_KEY, serializable);
	}

	private CacheTool getCacheTool() {
		return LiferayAbstractionFactorySupplier.getInstance().getCacheTool();
	}
}
