package de.mklinger.qetcher.liferay.client.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.security.KeyStore;
import java.util.Map;
import java.util.function.Supplier;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Modified;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.liferay.portal.configuration.metatype.bnd.util.ConfigurableUtil;
import com.liferay.portal.kernel.util.PropsKeys;
import com.liferay.portal.kernel.util.PropsUtil;

import de.mklinger.qetcher.client.InputConversionFile;
import de.mklinger.qetcher.client.InputJob;
import de.mklinger.qetcher.client.QetcherClient;
import de.mklinger.qetcher.client.QetcherClient.Builder;
import de.mklinger.qetcher.client.QetcherClientBuilders;
import de.mklinger.qetcher.client.common.QetcherException;
import de.mklinger.qetcher.client.httpclient.BodyProviders;
import de.mklinger.qetcher.liferay.client.QetcherClientService;
import de.mklinger.qetcher.liferay.client.common.KeyStores;
import de.mklinger.qetcher.liferay.client.common.VisibleForTesting;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
@Component(configurationPid = QetcherConfiguration.ID, immediate=true)
public class QetcherClientServiceImpl implements QetcherClientService {
	private static final Logger LOG = LoggerFactory.getLogger(QetcherClientServiceImpl.class);

	private QetcherConfiguration configuration;
	private QetcherClient client;

	@Activate
	@Modified
	public void activate(final Map<String, Object> properties) {
		closeClient();

		configuration = ConfigurableUtil.createConfigurable(QetcherConfiguration.class, properties);

		try {
			client = newClient();
			LOG.info("Successfully created new Qetcher client");
		} catch (final Exception e) {
			// TODO Use defunct wrapper?
			LOG.error("Error creating Qetcher client", e);
		}
	}

	@Deactivate
	public void closeClient() {
		if (client != null) {
			final QetcherClient oldClient = client;
			client = null;
			oldClient.close();
		}
	}

	private QetcherClient newClient() {
		final String[] serviceAddresses = configuration.serviceAddresses();
		if (serviceAddresses == null || serviceAddresses.length == 0) {
			throw new QetcherException("No Qetcher service addresses configured");
		}

		final KeyStore keyStore = loadKeyStore();
		final String keyPassword = emptyToNull(configuration.keyPassword());
		final KeyStore trustStore = loadTrustStore();

		return newClientBuilder()
				.serviceAddresses(serviceAddresses)
				.keyStore(keyStore, keyPassword)
				.trustStore(trustStore)
				.build();
	}

	@VisibleForTesting
	protected Builder newClientBuilder() {
		return QetcherClientBuilders.client();
	}

	private KeyStore loadKeyStore() {
		final String keyStoreLocation = getKeyStoreLocation();
		final String keyStorePassword = emptyToNull(configuration.keyStorePassword());
		final String keyStoreType = emptyToNull(configuration.keyStoreType());

		final KeyStore keyStore = KeyStores.load(
				keyStoreLocation,
				keyStorePassword,
				keyStoreType);
		return keyStore;
	}

	private String getKeyStoreLocation() {
		String keyStoreLocation = configuration.keyStoreLocation();
		if (keyStoreLocation == null || keyStoreLocation.isEmpty()) {
			keyStoreLocation = PropsUtil.get(PropsKeys.LIFERAY_HOME) + "/data/qetcher/qetcher-key.p12";
			LOG.info("No Qetcher key store location configured. Using default location: {}", keyStoreLocation);
		}
		return keyStoreLocation;
	}

	@VisibleForTesting
	protected KeyStore loadTrustStore() {
		KeyStore trustStore;
		try {
			trustStore = KeyStores.loadPemCertificates(
					getCaCertLocation(),
					getClass().getClassLoader());
		} catch (final IOException e) {
			throw new QetcherException("Error loading Qetcher ca certificate", e);
		}
		return trustStore;
	}

	private String getCaCertLocation() {
		return "classpath:"
				+ getClass().getPackage().getName().replace('.', '/')
				+ "/mklinger-ca-cert.pem";
	}

	private String emptyToNull(final String s) {
		if (s != null && s.trim().isEmpty()) {
			return null;
		}
		return s;
	}

	@Override
	public QetcherClient client() {
		if (client == null) {
			throw new QetcherException("Qetcher client is not initialized. Check configuration.");
		}
		return client;
	}

	@Override
	public InputConversionFile.Builder inputFileFor(final File inputFile) {
		return QetcherClientBuilders.inputFileFor(inputFile);
	}

	@Override
	public InputConversionFile.Builder inputFileFor(final Path inputFile) {
		return QetcherClientBuilders.inputFileFor(inputFile);
	}

	@Override
	public InputConversionFile.Builder inputFileFor(final Supplier<InputStream> inputStreamSupplier) {
		return QetcherClientBuilders.inputFile()
				.bodyProvider(BodyProviders.fromInputStream(inputStreamSupplier));
	}

	@Override
	public InputJob.Builder job() {
		return QetcherClientBuilders.job();
	}
}
