package de.mklinger.qetcher.liferay.client.common;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class Strings {
	/**
	 * Check whether the given CharSequence has actual text.
	 * More specifically, returns <code>true</code> if the string not <code>null</code>,
	 * its length is greater than 0, and it contains at least one non-whitespace character.
	 * <p>
	 *
	 * <pre>
	 * StringUtils.hasText(null) = false
	 * StringUtils.hasText("") = false
	 * StringUtils.hasText(" ") = false
	 * StringUtils.hasText("12345") = true
	 * StringUtils.hasText(" 12345 ") = true
	 * </pre>
	 *
	 * @param str the CharSequence to check (may be <code>null</code>)
	 * @return <code>true</code> if the CharSequence is not <code>null</code>,
	 * its length is greater than 0, and it does not contain whitespace only
	 * @see java.lang.Character#isWhitespace
	 */
	public static <T extends CharSequence> boolean hasText(final T str) {
		if (!hasLength(str)) {
			return false;
		}
		final int strLen = str.length();
		for (int i = 0; i < strLen; i++) {
			if (!Character.isWhitespace(str.charAt(i))) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Check that the given CharSequence is neither <code>null</code> nor of length 0.
	 * Note: Will return <code>true</code> for a CharSequence that purely consists of whitespace.
	 * <p>
	 *
	 * <pre>
	 * StringUtils.hasLength(null) = false
	 * StringUtils.hasLength("") = false
	 * StringUtils.hasLength(" ") = true
	 * StringUtils.hasLength("Hello") = true
	 * </pre>
	 *
	 * @param str the CharSequence to check (may be <code>null</code>)
	 * @return <code>true</code> if the CharSequence is not null and has length
	 * @see #hasText(String)
	 */
	private static <T extends CharSequence> boolean hasLength(final T str) {
		return str != null && str.length() > 0;
	}

	private static <T extends CharSequence> boolean isEmpty(final T str) {
		return str == null || str.length() == 0;
	}

	public static <T extends CharSequence> T requireText(final T str) {
		if (!hasText(str)) {
			throw new IllegalArgumentException();
		}
		return str;
	}

	public static <T extends CharSequence> T requireText(final T str, final String msg) {
		if (!hasText(str)) {
			throw new IllegalArgumentException(msg);
		}
		return str;
	}

	public static <T extends CharSequence> T requireNonEmpty(final T str) {
		if (isEmpty(str)) {
			throw new IllegalArgumentException();
		}
		return str;
	}

	public static <T extends CharSequence> T requireNonEmpty(final T str, final String msg) {
		if (isEmpty(str)) {
			throw new IllegalArgumentException(msg);
		}
		return str;
	}
}
