/*
 * Copyright 2016-present mklinger GmbH - http://www.mklinger.de
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.mklinger.qetcher.liferay.client.common;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableEntryException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.apache.commons.io.IOUtils;

import de.mklinger.qetcher.client.common.QetcherException;
import de.mklinger.qetcher.client.common.annotations.Nullable;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de - klingerm
 */
public class KeyStores {
	private static final String CLASSPATH_PREFIX = "classpath:";

	/** No instantiation */
	private KeyStores() {}

	public static KeyStore load(final String location, @Nullable final String password) {
		return load(location, password, KeyStore.getDefaultType(), getDefaultClassLoader());
	}

	public static KeyStore load(final String location, @Nullable final String password, final ClassLoader classLoader) {
		return load(location, password, KeyStore.getDefaultType(), classLoader);
	}

	public static KeyStore load(final String location, @Nullable final String password, final String type) {
		return load(location, password, type, getDefaultClassLoader());
	}

	private static ClassLoader getDefaultClassLoader() {
		return KeyStores.class.getClassLoader();
	}

	public static KeyStore load(final String location, @Nullable final String password, final String type, final ClassLoader classLoader) {
		Objects.requireNonNull(location);
		Objects.requireNonNull(type);
		Objects.requireNonNull(classLoader);
		try {
			final KeyStore keyStore = KeyStore.getInstance(type);
			try(InputStream in = newInputStream(location, classLoader)) {
				keyStore.load(in, toCharArray(password));
			}
			return keyStore;
		} catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
			throw new QetcherException("Error loading keystore from location '{}'", location, e);
		}
	}

	public static KeyStore loadPemCertificates(final String location, final ClassLoader classLoader) throws IOException {
		try (InputStream in = newInputStream(location, classLoader)) {
			return loadPemCertificates(in);
		}
	}

	private static KeyStore loadPemCertificates(final InputStream in) throws IOException {
		try {
			return doLoadPemCertificates(in);
		} catch (CertificateException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException e) {
			throw new QetcherException("Error loading certificates", e);
		}
	}

	private static KeyStore doLoadPemCertificates(final InputStream in) throws IOException, CertificateException, KeyStoreException, NoSuchAlgorithmException, UnrecoverableEntryException {
		InputStream actualIn;
		if (in.markSupported()) {
			actualIn = in;
		} else {
			final ByteArrayOutputStream bout = new ByteArrayOutputStream();
			IOUtils.copy(in, bout);
			actualIn = new ByteArrayInputStream(bout.toByteArray());
		}

		final List<Certificate> certificates = new ArrayList<>();
		final CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
		CertificateException error = null;

		while (true) {
			final Certificate certificate;
			try {
				certificate = certFactory.generateCertificate(actualIn);
				if (certificate == null) {
					break;
				}
				certificates.add(certificate);
			} catch (final CertificateException e) {
				if (error == null) {
					error = e;
				} else {
					error.addSuppressed(e);
				}
				// Ignore. Happens when no more certificates are available in the stream
				break;
			}
		}

		if (certificates.isEmpty()) {
			throw new CertificateException("Unable to load certificates from stream", error);
		}

		return createKeyStoreWith(certificates);
	}

	private static KeyStore createKeyStoreWith(final List<Certificate> certificates) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException {
		final KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
		keyStore.load(null, null);

		for (int i = 0; i < certificates.size(); i++) {
			final Certificate c = certificates.get(i);
			keyStore.setCertificateEntry("cert" + i, c);
		}

		return keyStore;
	}

	private static InputStream newInputStream(final String location, final ClassLoader classLoader) throws IOException {
		if (location.startsWith(CLASSPATH_PREFIX)) {
			final String classpathLocation = location.substring(CLASSPATH_PREFIX.length());
			InputStream in = classLoader.getResourceAsStream(classpathLocation);
			if (in == null) {
				in = Thread.currentThread().getContextClassLoader().getResourceAsStream(classpathLocation);
			}
			if (in == null) {
				throw new FileNotFoundException("Classpath resource not found: " + classpathLocation);
			}
			return in;
		} else if (location.startsWith("/") || location.startsWith("./")) {
			return new FileInputStream(location);
		} else {
			return URI.create(location).toURL().openStream();
		}
	}

	private static char[] toCharArray(final String password) {
		if (password == null) {
			return null;
		}
		return password.toCharArray();
	}
}
