package de.mklinger.qetcher.liferay.client.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.mklinger.qetcher.client.InputConversionFile;
import de.mklinger.qetcher.client.InputJob;
import de.mklinger.qetcher.client.QetcherClientException;
import de.mklinger.qetcher.client.QetcherRemoteException;
import de.mklinger.qetcher.client.common.concurrent.Delay;
import de.mklinger.qetcher.client.model.v1.Job;
import de.mklinger.qetcher.client.model.v1.JobState;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.client.model.v1.MediaTypes;
import de.mklinger.qetcher.liferay.client.QetcherClientService;
import de.mklinger.qetcher.liferay.client.QetcherService;
import de.mklinger.qetcher.liferay.client.impl.abstraction.liferay71.LiferayAbstractionFactory;
import de.mklinger.qetcher.liferay.client.impl.htmlinliner.HtmlElementInliner;
import de.mklinger.qetcher.liferay.client.impl.htmlinliner.QetcherHtmlInlineConfiguration;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
@Component
public class QetcherServiceImpl implements QetcherService {
	private static final int JOB_WAIT_TIMEOUT_MINUTES = 5;

	private static final Logger LOG = LoggerFactory.getLogger(QetcherServiceImpl.class);

	@Reference
	private QetcherClientService clientService;

	@Override
	public Path convertToTempFile(final InputStream inputStream, final String fromMediaType, final String toMediaType) {
		final MediaType from = MediaType.valueOf(fromMediaType);
		final MediaType to = MediaType.valueOf(toMediaType);

		final InputStream actualInputStream = getActualInputStream(inputStream, from);

		final InputConversionFile inputFile = clientService.inputFileFor(() -> actualInputStream)
				.mediaType(from)
				.build();

		final InputJob inputJob = clientService.job()
				.fromFile(inputFile)
				.toMediaType(to)
				.build();

		final Job job = createJobWithTimeout(inputJob)
				.join();

		try {

			return getJobDoneWithTimeout(job)
					.thenApply(this::requireSuccess)
					.thenApply(this::getSingleResultFileId)
					.thenCompose(clientService.client()::downloadAsTempFile)
					.join();

		} finally {
			try {
				deleteJobWithTimeout(job)
				.join();
			} catch (final Exception e) {
				LOG.warn("Error deleting job", e);
			}
		}

	}

	private InputStream getActualInputStream(final InputStream inputStream, final MediaType from) {
		if (MediaTypes.HTML.isCompatible(from)) {
			return inlineHtmlElements(inputStream);
		} else {
			return inputStream;
		}
	}

	private InputStream inlineHtmlElements(final InputStream inputStream) {
		LOG.info("Inlining HTML elements...");

		final String baseUri = LiferayAbstractionFactory.getInstance().getPortalTool().getBaseUrl();
		if (baseUri == null) {
			LOG.warn("Could not get base URL - no HTML inlining can be done");
			return inputStream;
		}

		byte[] data;

		try (InputStream in = inputStream) {
			try (HtmlElementInliner inliner = new HtmlElementInliner(new QetcherHtmlInlineConfiguration())) {
				data = inliner.inline(in, baseUri);
			}
		} catch (final IOException e) {
			throw new UncheckedIOException(e);
		}

		return new ByteArrayInputStream(data);
	}

	private CompletableFuture<Job> createJobWithTimeout(final InputJob inputJob) {
		return Delay.timeout(
				clientService.client().createJob(inputJob),
				10, TimeUnit.SECONDS,
				() -> new TimeoutException(
						"Timeout creating job "
								+ inputJob.getInputConversionFile().getFilename()
								+ " "
								+ inputJob.getFromMediaType()
								+ " -> "
								+ inputJob.getToMediaType()));
	}

	private CompletableFuture<Job> getJobDoneWithTimeout(final Job job) {
		return Delay.timeout(
				clientService.client().getJobDone(job),
				JOB_WAIT_TIMEOUT_MINUTES, TimeUnit.MINUTES,
				() -> new TimeoutException(
						"Timeout getting job done "
								+ job.getJobId()
								+ " "
								+ job.getFromMediaType()
								+ " -> "
								+ job.getToMediaType()));
	}

	private Job requireSuccess(final Job job) {
		if (job.getState() != JobState.SUCCESS) {
			Exception cause;
			if (job.getError() != null) {
				cause = new QetcherRemoteException(job.getError(), -1);
			} else {
				cause = null;
			}
			throw new QetcherClientException("Job " + job.getJobId() + " is not in state SUCCESS but " + job.getState(), cause);
		}

		return job;
	}

	private String getSingleResultFileId(final Job job) {
		final List<String> resultFileIds = job.getResultFileIds();
		if (resultFileIds == null || resultFileIds.size() != 1) {
			throw new QetcherClientException("Expected 1 result file but was " + resultFileIds.size());
		}
		return resultFileIds.get(0);
	}

	private CompletableFuture<Void> deleteJobWithTimeout(final Job job) {
		return Delay.timeout(
				clientService.client().deleteJob(job),
				30, TimeUnit.SECONDS,
				() -> new TimeoutException(
						"Timeout deleting job "
								+ job.getJobId()
								+ " "
								+ job.getFromMediaType()
								+ " -> "
								+ job.getToMediaType()));
	}
}
