/**
 *                                 Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2018 Mike Hummel
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package de.mhus.lib.vaadin.desktop;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.Locale;
import java.util.TreeMap;

import com.vaadin.server.FontAwesome;
import com.vaadin.ui.AbstractComponent;
import com.vaadin.ui.Button.ClickEvent;
import com.vaadin.ui.Component;
import com.vaadin.ui.CssLayout;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Label;
import com.vaadin.ui.MenuBar;
import com.vaadin.ui.MenuBar.MenuItem;
import com.vaadin.ui.NativeButton;
import com.vaadin.ui.UI;
import com.vaadin.ui.VerticalLayout;

import de.mhus.lib.core.MString;
import de.mhus.lib.core.logging.Log;
import de.mhus.lib.core.util.MNls;
import de.mhus.lib.core.util.MNlsProvider;

public class Desktop extends CssLayout implements MNlsProvider {

	private static final long serialVersionUID = 1L;
	private MenuBar menuBar;
	private MenuItem menuSpaces;
	private HorizontalLayout contentScreen;
	private MenuItem[] menuSpace = new MenuItem[4];
	private MenuItem menuLeave;
	protected MenuItem menuUser;
	private MenuItem menuLogout;
	protected GuiSpaceService currentSpace;
	private MenuItem menuOverview;
	private CssLayout overView;
	private MenuItem menuHistory;
	private MenuItem menuBack;
	private static Log log = Log.getLog(Desktop.class);
	private LinkedList<String> history = new LinkedList<>();
	private TreeMap<String,GuiSpaceService> spaceList = new TreeMap<String, GuiSpaceService>();
	private HashMap<String, AbstractComponent> spaceInstanceList = new HashMap<String, AbstractComponent>();
	private HashMap<String, HelpContext> helpInstanceList = new HashMap<String, HelpContext>();
	private GuiApi api;
	private MNls nls;
	private int tileWidth = 200;
	private int tileHeight = 160;
	private int tileHorizontalGap = 20;
	private MenuItem menuHelp;
	private boolean helpActive;
	private VerticalLayout helpView;

	public Desktop(GuiApi api) {
		this.api = api;
		initGui();
	}

	protected void initGui() {
		
		helpView = new VerticalLayout();
		helpView.setSizeFull();
		helpView.setStyleName("help-panel");
		
		overView = new CssLayout();
		overView.setSizeFull();
		overView.setStyleName("overview");
		
		menuBar = new MenuBar();
		menuSpaces = menuBar.addItem(MNls.find(this, "menu.spaces=Spaces"), null);

		menuHistory = menuBar.addItem(MNls.find(this, "menu.history=History"), null);
		menuBack = menuHistory.addItem(MNls.find(this, "menu.back=Back"), new MenuBar.Command() {

			@Override
			public void menuSelected(MenuItem selectedItem) {
				navigateBack();
			}
			
		});
		menuHistory.addSeparator();
		for (int i = 0; i < 15; i++)
			menuHistory.addItem("", new MenuBar.Command() {

				@Override
				public void menuSelected(MenuItem selectedItem) {
					String text = selectedItem.getDescription();
					if (MString.isSet(text)) {
						String[] parts = text.split("\\|", 4);
						if (parts.length == 4) {
							if (parts[2].equals("null")) parts[2] = null;
							if (parts[3].equals("null")) parts[3] = null;
							openSpace(parts[1], parts[2], parts[3]);
						}
					}
				}
				
			});
		
		menuSpace[0] = menuBar.addItem("", null);
		menuSpace[1] = menuBar.addItem("", null);
		menuSpace[2] = menuBar.addItem("", null);
		menuSpace[3] = menuBar.addItem("", null);
		
		menuUser = menuBar.addItem( "?", null);
		menuUser.setStyleName("right");
		menuLogout = menuUser.addItem(MNls.find(this, "menu.logout=Logout"), new MenuBar.Command() {
			
			@Override
			public void menuSelected(MenuItem selectedItem) {
				try {
					getApi().getAccessControl().signOut();
				} catch (Throwable t) {
					log.d(t);
				}
				try {
					UI.getCurrent().close();
				} catch (Throwable t) {
					log.d(t);
				}
				UI.getCurrent().getPage().reload();
			}
		});
		menuUser.addSeparator();
		menuHelp = menuUser.addItem(MNls.find(this, "menu.help=Help"), new MenuBar.Command() {
			@Override
			public void menuSelected(MenuItem selectedItem) {
				doShowHelp();
			}
		});
		menuHelp.setEnabled(false);
		menuUser.addSeparator();
		
		setStyleName("desktop-screen");
		menuBar.setStyleName("menubar");
		
		addComponent(menuBar);
		
		contentScreen = new HorizontalLayout();
		contentScreen.addStyleName("content");
		contentScreen.setSizeFull();
		addComponent(contentScreen);
		setSizeFull();
		
		showOverview(false);
	}

	protected void doShowHelp() {
		if (currentSpace == null) return;
		
		if (helpActive) {
			hideHelp();
			return;
		}
		
		showHelpTopic(null);
	}

	public void hideHelp() {
		synchronized (this) {
			if (!helpActive) return;
			helpView.removeAllComponents();
			contentScreen.removeComponent(helpView);
			helpActive = false;
		}
	}
	
	public void showHelpTopic(String topic) {
		
		HelpContext help = getHelpContext(currentSpace.getName());
		if (help == null) {
			menuHelp.setEnabled(false);
			return;
		}
		Component component = help.getComponent();
		if (component == null) return;

		synchronized (this) {
			if (!helpActive) {
				contentScreen.addComponent(helpView);
				contentScreen.setExpandRatio(helpView, 0.5f);
			};
			helpView.removeAllComponents();
			helpView.addComponent(component);
			
		}
		helpActive = true;
	}
	
	
	public void refreshSpaceList() {
		
		String name = "?";
		try {
			name = getApi().getAccessControl().getAccount().getDisplayName();
		} catch (Throwable t) {
			log.t(t);
		}
		menuUser.setText(name == null ? "?" : name);
		
		menuSpaces.removeChildren();
		overView.removeAllComponents();
		
		menuOverview = menuSpaces.addItem(MNls.find(this, "menu.overview=Overview"), new MenuBar.Command() {
			
			@Override
			public void menuSelected(MenuItem selectedItem) {
				showOverview(true);
			}
		});

		menuSpaces.addSeparator();
		
		LinkedList<GuiSpaceService> componentList = new LinkedList<>();
		for (GuiSpaceService space : spaceList.values()) {
			
			try {
				if (space.isHiddenSpace() || !hasAccess(space) || !space.hasAccess(getApi().getAccessControl())) continue;
				componentList.add(space);
			} catch (Throwable t) {
				log.d(space,t);
			}
		}

		Locale locale = UI.getCurrent().getPage().getWebBrowser().getLocale();

		for (final GuiSpaceService space : componentList ) {
			
			AbstractComponent tile = space.createTile();
			if (tile == null) {
				NativeButton button = new NativeButton();
				button.setHtmlContentAllowed(false);
				button.setCaption( space.getDisplayName(locale));
				button.addClickListener(new NativeButton.ClickListener() {
					
					@Override
					public void buttonClick(ClickEvent event) {
						openSpace(space.getName(), null, null); // not directly to support history
					}
				});
				tile = button;
				tile.addStyleName("cursorhand");
			}
			int tileSize = space.getTileSize();
			if (tileSize < 1) tileSize = 1;
			if (tileSize > 3) tileSize = 3;
			tile.addStyleName("thumbnail" + tileSize);
			tile.setWidth(( tileWidth * tileSize + (tileSize - 1) * tileHorizontalGap ) + "px");
			tile.setHeight( tileHeight + "px");
			overView.addComponent(tile);
			
			if (!space.isHiddenInMenu()) {
				MenuItem item = menuSpaces.addItem(space.getDisplayName(locale), new MenuBar.Command() {
					
					@Override
					public void menuSelected(MenuItem selectedItem) {
						openSpace(space.getName(), null, null); // not directly to support history
					}
				});
				item.setEnabled(true);
			}
		}
		
		if (componentList.size() > 0)
			menuSpaces.addSeparator();
		
		menuLeave = menuSpaces.addItem(MNls.find(this, "menu.leave=Leave Space"), new MenuBar.Command() {
			
			@Override
			public void menuSelected(MenuItem selectedItem) {
				if (currentSpace == null) return;
				removeSpaceComponent(currentSpace.getName());
				currentSpace = null;
				showOverview(true);
			}
		});
		menuLeave.setEnabled(false);

	}

	public void removeSpaceComponent(String name) {
		AbstractComponent c = spaceInstanceList.remove(name);
		if (c != null && c instanceof GuiLifecycle) ((GuiLifecycle)c).doDestroy();
	}

	protected boolean hasAccess(GuiSpaceService space) {
		return getApi().hasAccess(space.getName());
	}

	protected String showSpace(GuiSpaceService space, String subSpace, String search) {
		boolean exists = spaceInstanceList.containsKey(space.getName());
		AbstractComponent component = getSpaceComponent(space.getName());
		HelpContext help = getHelpContext(space.getName());
		
		menuHelp.setEnabled(false);
		helpActive = false;
		
		contentScreen.removeAllComponents();
		cleanupMenu();
		if (component == null) {
			contentScreen.addComponent(new Label(MNls.find(this, "spaces.unavailable=Space currently not available") + space.getName()));
			addComponent(contentScreen);
			return null;
		}
		
		component.setSizeFull();
		contentScreen.addComponent(component);
		contentScreen.setExpandRatio(component, 1f);
		
		Locale locale = UI.getCurrent().getPage().getWebBrowser().getLocale();

		menuHistory.setText(space.getDisplayName(locale));
		menuLeave.setEnabled(true);
		currentSpace = space;
		space.createMenu(component,menuSpace);
		
		if (help != null) {
			menuHelp.setEnabled(true);
		}
		
		if (component instanceof Navigable) {
			if ( (MString.isSet(subSpace) || MString.isSet(search))) 
				return ((Navigable)component).navigateTo(subSpace, search);
			else
				((Navigable)component).onShowSpace(!exists);
		}
		return space.getDisplayName(locale);
	}

	protected void showOverview(boolean setLinking) {
		if (menuLeave != null) menuLeave.setEnabled(false);
		contentScreen.removeAllComponents();
		cleanupMenu();
		currentSpace = null;
		contentScreen.addComponent(overView);
		contentScreen.setExpandRatio(overView, 1f);

		menuHistory.setText(MNls.find(this, "menu.history=History"));
		if (setLinking)
			UI.getCurrent().getPage().setUriFragment("");
	}

	private void cleanupMenu() {
		
		for (int i=0; i < menuSpace.length; i++) {
			menuSpace[i].removeChildren();
			menuSpace[i].setText("");
			menuSpace[i].setVisible(false);
		}
	}

	public void doUpdateHistoryMenu() {
		int cnt = -2;
		for (MenuItem c : menuHistory.getChildren()) {
			if (cnt >= 0) {
				if (history.size() - cnt - 1 < 0) {
					c.setText("");
					c.setDescription("");
					c.setIcon(null);
				} else {
					String x = history.get(history.size() - cnt - 1);
					c.setText(MString.beforeIndex(x, '|'));
					c.setDescription(x);
					c.setIcon(FontAwesome.ARROW_RIGHT);
				}
			}
			cnt++;
		}
	}

	public void rememberNavigation(String caption, String space, String subSpace, String search, boolean navLink) {
		String newEntry = caption.replace('|', ' ') + "|" + space + "|" + subSpace + "|" + search;
		while (this.history.remove(newEntry) ) {} // move up
		this.history.add(newEntry);
		doUpdateHistoryMenu();
		if (navLink)
			UI.getCurrent().getPage().setUriFragment("!:" + space + "/" + (subSpace == null ? "" : subSpace) + "/" + (search == null ? "" : search));
	}

	public boolean openSpace(String spaceId, String subSpace, String search) {
		return openSpace(spaceId, subSpace, search, true, true);
	}
	
	public boolean openSpace(String spaceId, String subSpace, String search, boolean history, boolean navLink) {
		GuiSpaceService space = getSpace(spaceId);
		if (space == null) return false;
		if (!getApi().hasAccess(space.getName()) || !space.hasAccess(getApi().getAccessControl())) return false;

		String ret = showSpace(space, subSpace, search);
		if (ret != null && history) {
			String newEntry = ret.replace('|', ' ') + "|" + spaceId + "|" + subSpace + "|" + search;
			while (this.history.remove(newEntry) ) {} // move up
			this.history.add(newEntry);
			doUpdateHistoryMenu();
		}
		if (navLink)
			UI.getCurrent().getPage().setUriFragment("!:" + spaceId + "/" + (subSpace == null ? "" : subSpace) + "/" + (search == null ? "" : search));
		return ret != null;
	}

	public void navigateBack() {
		if (history.size() == 0) return;
		String link = history.removeLast();
		if (history.size() == 0) return;
		link = history.getLast();
		doUpdateHistoryMenu();
		String[] parts = link.split("\\|", 4);
		if (parts[2].equals("null")) parts[2] = null;
		if (parts[3].equals("null")) parts[3] = null;
		openSpace(parts[1], parts[2], parts[3], false, true);
	}

	public GuiSpaceService getSpace(String name) {
		return spaceList.get(name);
	}

	public GuiApi getApi() {
		return api;
	}

	public void close() {
		spaceList.clear();
		for (AbstractComponent v : spaceInstanceList.values())
			if (v instanceof GuiLifecycle) ((GuiLifecycle)v).doDestroy();
		spaceInstanceList.clear();
	}

	public void addSpace(GuiSpaceService service) {
		log.d("add space", service);
		spaceList.put(service.getName(),service);
		refreshSpaceList();
	}

	public void removeSpace(GuiSpaceService service) {
		log.d("remove space",service);
		spaceList.remove(service.getName());
		AbstractComponent v = spaceInstanceList.remove(service.getName());
		if (v != null && v instanceof GuiLifecycle) ((GuiLifecycle)v).doDestroy();
		refreshSpaceList();
	}
	
	public AbstractComponent getSpaceComponent(String name) {
		GuiSpaceService space = spaceList.get(name);
		if (space == null) return null;
		AbstractComponent instance = spaceInstanceList.get(name);
		if (instance == null) {
			instance = space.createSpace();
			if (instance == null) return null;
			if (instance instanceof GuiLifecycle) ((GuiLifecycle)instance).doInitialize();
			spaceInstanceList.put(name, instance);
		}
		return instance;
	}

	public HelpContext getHelpContext(String name) {
		GuiSpaceService space = spaceList.get(name);
		if (space == null) return null;
		HelpContext instance = helpInstanceList.get(name);
		if (instance == null) {
			Locale locale = UI.getCurrent().getPage().getWebBrowser().getLocale();
			instance = space.createHelpContext(locale);
			if (instance == null) return null;
			if (instance instanceof GuiLifecycle) ((GuiLifecycle)instance).doInitialize();
			helpInstanceList.put(name, instance);
		}
		return instance;
	}
	
	@Override
	public MNls getNls() {
		if (nls == null) nls = MNls.lookup(this);
		return nls;
	}

	public int getTileWidth() {
		return tileWidth;
	}

	public void setTileWidth(int tileWidth) {
		this.tileWidth = tileWidth;
	}

	public int getTileHeight() {
		return tileHeight;
	}

	public void setTileHeight(int tileHeight) {
		this.tileHeight = tileHeight;
	}

	public int getTileHorizontalGap() {
		return tileHorizontalGap;
	}

	public void setTileHorizontalGap(int tileHorizontalGap) {
		this.tileHorizontalGap = tileHorizontalGap;
	}

}
