/**
 *                                 Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2018 Mike Hummel
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package de.mhus.lib.vaadin;

import java.util.LinkedList;
import java.util.List;

import com.vaadin.data.Property;
import com.vaadin.data.Property.ValueChangeEvent;
import com.vaadin.event.ItemClickEvent;
import com.vaadin.event.ItemClickEvent.ItemClickListener;
import com.vaadin.ui.AbstractComponent;
import com.vaadin.ui.Button;
import com.vaadin.ui.Button.ClickEvent;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Notification;
import com.vaadin.ui.Panel;
import com.vaadin.ui.Table;
import com.vaadin.ui.Table.HeaderClickEvent;
import com.vaadin.ui.UI;
import com.vaadin.ui.VerticalLayout;
import com.vaadin.ui.VerticalSplitPanel;

import de.mhus.lib.annotations.pojo.Hidden;
import de.mhus.lib.core.ILog;
import de.mhus.lib.core.MActivator;
import de.mhus.lib.core.MString;
import de.mhus.lib.core.logging.Log;
import de.mhus.lib.core.util.FilterRequest;
import de.mhus.lib.core.util.MNls;
import de.mhus.lib.core.util.MNlsBundle;
import de.mhus.lib.core.util.MNlsProvider;
import de.mhus.lib.vaadin.form.VaadinPojoForm;

public abstract class AbstractListEditor<E> extends VerticalLayout implements MNlsProvider, ILog {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	protected static final Object MY_NEW_MARKER = new Object();

	public static final String LABEL_SAVED_NEW = "entity.new=new";
	public static final String LABEL_SAVED = "entity.saved=saved";
	public static final String LABEL_DELETED = "entity.deleted=deleted";
	public static final String LABEL_CANELED = "entity.canceled=canceled";

	protected SimpleTable table;
	private Button bNew;
	private Button bUpdate;
	private Button bDelete;
	protected Object editMode;
	protected VaadinPojoForm<E> model;
	protected SearchField filter;
	private boolean showSearchField = true;
	private boolean needSortUpdate = false;
	private Panel detailsPanel;
	private boolean fullSize;
	private VerticalLayout detailsPanelContent;
	private MNlsBundle nlsBundle;
	private boolean modified = false;
	private boolean initialized = false;

	@Hidden
	private Log log = Log.getLog(this);
	private String sortedColumn;
	private boolean sortedAscending;
	
	@SuppressWarnings("serial")
	public void initUI() {
		if (isInitialized()) return;
		setInitialized(true);
		
    	if (fullSize) setSizeFull();
		setSpacing(true);
		setMargin(true);
		
		filter = new SearchField(this);
		filter.setListener(new SearchField.Listener() {
			
			@Override
			public void doFilter(SearchField searchField) {
				AbstractListEditor.this.doFilter();
			}
		});
		table = new SimpleTable(getTableName());
		table.setSelectable(true);
        table.setMultiSelect(false);
        table.setImmediate(true);
        table.setSizeFull();
        table.createDataSource(createColumnDefinitions());
        fillDataSource(new FilterRequest(""));
                
        table.addValueChangeListener(new Property.ValueChangeListener() {
			
			@Override
			public void valueChange(ValueChangeEvent event) {
				doSelectionChanged();
			}
		});

        table.addItemClickListener(new ItemClickListener() {
			
			@Override
			public void itemClick(ItemClickEvent event) {
				doItemClick(event);
			}
		});
        
        table.addHeaderClickListener(new Table.HeaderClickListener() {
			
			@Override
			public void headerClick(HeaderClickEvent event) {
				
				String name = String.valueOf( event.getPropertyId() );
				if (name.equals(sortedColumn))
					sortedAscending = ! sortedAscending;
				else
					sortedAscending = true;
				sortedColumn = name;
				if (needSortUpdate)
					updateDataSource();
			}
		});
        
        detailsPanel = new Panel(getDetailsName());
        detailsPanel.setWidth("100%");
        detailsPanelContent = new VerticalLayout();
        detailsPanel.setContent(detailsPanelContent);
    	if (fullSize) detailsPanel.setSizeFull();
    	// detailsPanel.setScrollable(false);
        
        try {
        	
	        model = createForm();
	        if (model.getForm().getNlsBundle() == null)
	        	model.getForm().setNlsBundle(MNlsBundle.lookup(this));
//	        model.doBuild(getActivator());
	        model.doBuild();
	        detailsPanelContent.addComponent(model);
        } catch (Exception e) {
        	e.printStackTrace();
        }   
        
        

        HorizontalLayout buttonBar = new HorizontalLayout();
        
        bNew = new Button(MNls.find(this, "button.create=Create"));
        buttonBar.addComponent(bNew);
        bNew.addClickListener(new ClickListener() {
			
			@Override
			public void buttonClick(ClickEvent event) {
				doNew();
			}
		});

        bUpdate = new Button(MNls.find(this, "button.edit=Edit"));
        buttonBar.addComponent(bUpdate);
        bUpdate.addClickListener(new ClickListener() {
			
			@Override
			public void buttonClick(ClickEvent event) {
				doUpdate();
			}
		});

        bDelete = new Button(MNls.find(this, "button.delete=Delete"));
        buttonBar.addComponent(bDelete);
        bDelete.addClickListener(new ClickListener() {
			
			@Override
			public void buttonClick(ClickEvent event) {
				doDelete();
			}
		});

        
        editMode = null;
        
        createCustomButtons(buttonBar);
        composeElements(filter, table, detailsPanel, buttonBar );
        
        updateEnabled();

	}

	protected void doItemClick(ItemClickEvent event) {
		if (editMode == null && event.isDoubleClick())
			doUpdate();
	}

	protected MActivator getActivator() {
		return null;
	}

	protected void createCustomButtons(HorizontalLayout buttonBar) {
	}

	protected String getDetailsName() {
		return MNls.find(this, "panel.details.caption=Details");
	}

	protected void doFilter() {
		updateDataSource();
	}

	protected void composeElements(AbstractComponent filter, AbstractComponent table, AbstractComponent formPanel, AbstractComponent buttonBar) {
		
		if (showSearchField) addComponent(filter);
		
//		addComponent(table);
//		setExpandRatio(table, 1.0f);
//		
//		addComponent(formPanel);
//		addComponent(buttonBar);
		
		VerticalSplitPanel split = new VerticalSplitPanel();
				
		split.setFirstComponent(table);
		table.setSizeUndefined();
		table.setSizeFull();
//		split.setSplitPosition(200);
		split.setSecondComponent(formPanel);
		split.setSizeFull();
//		split.setHeight("300px");
		addComponent(split);
		setExpandRatio(split, 1.0f);
		addComponent(buttonBar);
		setSizeFull();
	}

	protected abstract ColumnDefinition[] createColumnDefinitions();
	
	protected String getTableName() {
		return MNls.find(this, "panel.table.caption=Table");
	};
	
	protected abstract E createTarget();
	
	protected VaadinPojoForm createForm() {
		try {
			VaadinPojoForm form = new VaadinPojoForm(createTarget());
			form.setPojo(createTarget());
			return form;
		} catch (Throwable t) {
			log.w(t);
		}
		return null;
	}
	
	protected void doSelectionChanged() {
		if (editMode != null) return;
		Object selectedId = table.getValue();
		E target = null;
		if (selectedId == null)
			target = createTarget();
		else
			target = getTarget(selectedId);
		model.setPojo(target);
		updateEnabled();
	}

	protected void doDelete() {
		if (editMode != null) {
			// Cancel
			doCancel();
			return;
		}
		
		Object selectedId = table.getValue();
		if (selectedId == null || !canDelete(selectedId)) return;
		final E selectedObj = getTarget(selectedId);
		if (selectedObj == null) return;
		
		ConfirmDialog.show(getUI(), MNls.find(this, "confirm.delete=Delete"), MNls.find(this, "confirm.question=Are you sure?"),
				MNls.find(this, "confirm.yes=Yes"), MNls.find(this, "confirm.no=Cancel"), new ConfirmDialog.Listener() {

		            @Override
					public void onClose(ConfirmDialog dialog) {
		                if (dialog.isConfirmed()) {
		                	try {
			                	doDelete(selectedObj);
								showInformation(MNls.find(AbstractListEditor.this, LABEL_DELETED));
		                	} catch (Throwable e) {
		        				log().i(e);
		        				showError(e);
		                	}
		                	model.setPojo(createTarget());
	
		                	updateDataSource();
		                	modified = true;
		                } else {
		                }
		            }
		        });
		
	}

	protected void doCancel() {
		if (editMode == null) return;
		try {
			if (!MY_NEW_MARKER.equals(editMode)) {
				doCancel(getTarget(editMode));
				showInformation(MNls.find(AbstractListEditor.this,LABEL_CANELED));
			}
		} catch (Throwable t) {
			log().i(t);
			showError(t);
		}
		editMode = null;
    	model.setPojo(createTarget());
		updateEnabled();
	}

	protected abstract void doCancel(E entry) throws Exception;

	protected abstract void doDelete(E entry) throws Exception;
	
	protected void doUpdate() {
		if (editMode == null) {
			Object selectedId = table.getValue();
			if (selectedId == null || !canUpdate(selectedId)) return;
			try {
				doUpdate(selectedId);
			} catch (Exception e) {
				e.printStackTrace();
			}
			
			// fill 
		} else {
			// save
			try {
				if (!canUpdate(editMode)) return;
				@SuppressWarnings("unchecked")
				E entity = (E) model.getPojo();
				if (MY_NEW_MARKER.equals(editMode)) {
					String error = doValidateNew(entity);
					if (error != null) {
						showError(error);
						return;
					}
					doSaveNew(entity);
					showInformation(MNls.find(AbstractListEditor.this,LABEL_SAVED_NEW));
				} else {
					String error = doValidate(entity);
					if (error != null) {
						showError(error);
						return;
					}
					doSave(entity);
					showInformation(MNls.find(AbstractListEditor.this,LABEL_SAVED));
				}
				
				updateDataSource();
            	modified = true;
				
			} catch (Throwable e) {
				log().i(e);
				showError(e);
			}
			editMode = null;
		}
		
		updateEnabled();
		
	}

	/**
	 * Overwrite this function if you want to validate a new entity before saving. Return null if the
	 * values are correct or return the internationalized error message.
	 * 
	 * @param entity
	 * @return
	 */
	protected String doValidateNew(E entity) {
		return null;
	}

	/**
	 * Overwrite this function if you want to validate an updated entity before saving. Return null if the
	 * values are correct or return the internationalized error message.
	 * 
	 * @param entity
	 * @return
	 */
	protected String doValidate(E entity) {
		return null;
	}
	
	public void showError(Throwable e) {
		showError(e.getMessage());
	}

	public void showError(String e) {
		Notification.show(e, Notification.Type.ERROR_MESSAGE);
	}
	
	public void showInformation(String msg) {
		if (MString.isEmpty(msg)) return;
		Notification.show(msg, Notification.Type.HUMANIZED_MESSAGE);
	}
	
	protected void doSaveNew(E entity) throws Exception {
		doSave(entity);
	}

	protected void doUpdate(Object selectedId) {
		editMode = selectedId;
		if (MY_NEW_MARKER.equals(editMode)) return;

		E target = getEditableTarget(editMode);
		model.setPojo(target);
	}

	protected abstract void doSave(E entry) throws Exception;

	protected abstract E getEditableTarget(Object id);
	
	protected abstract E getTarget(Object id);

	protected void doNew() {
		if (editMode != null || !canNew()) return;
		editMode = MY_NEW_MARKER;
    	model.setPojo(createTarget());
		
		updateEnabled();
	}

	protected void updateEnabled() {
		Object selectedId = table.getValue();
		
		if (!isEditMode()) {
			bNew.setEnabled(canNew());
			bNew.setCaption(MNls.find(this, "button.create=Create"));
			bUpdate.setEnabled(selectedId != null && canUpdate(selectedId) );
			bUpdate.setCaption(MNls.find(this, "button.edit=Edit"));
			bDelete.setEnabled(selectedId != null && canDelete(selectedId));
			bDelete.setCaption(MNls.find(this, "button.delete=Delete"));
			if (model != null) model.setEnabled(false);
			table.setEnabled(true);
		} else {
			bNew.setEnabled(false);
			bNew.setCaption(MNls.find(this, "button.create=Create"));
			bUpdate.setEnabled(true);
			bUpdate.setCaption(MNls.find(this, "button.save=Set"));
			bDelete.setEnabled(true);
			bDelete.setCaption(MNls.find(this, "button.cancel=Cancel"));
			model.setEnabled(true);
			table.setEnabled(false);
		}
		doUpdateEnabled(selectedId);
	}
	
	/**
	 * Overwrite this to update your own buttons
	 */
	protected void doUpdateEnabled(Object selectedId) {
	}

	public boolean isEditMode() {
		return editMode != null;
	}

	public boolean canDelete(Object selectedId) {
		return true;
	}

	public boolean canUpdate(Object selectedId) {
		return true;
	}

	public boolean canNew() {
		return true;
	}

	protected void fillDataSource(FilterRequest filter) {
		try {
			table.removeAllItems();
			for ( E entity : createDataList(filter) ) {
				table.addRow(getId(entity), getValues(entity));
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	protected abstract Object[] getValues(E entry);

	protected abstract Object getId(E entry);

	protected abstract List<E> createDataList(FilterRequest filter);

	public void updateDataSource() {
		try {
			LinkedList<Object> newIds = new LinkedList<Object>();
			for ( E entity : createDataList(filter.createFilterRequest())) {
				Object id = getId(entity);
				Object[] values = getValues(entity);
				if (!table.updateRow(id, values))
					table.addRow(id, values);
				newIds.add(id);
			}
			
			for ( Object id : new LinkedList<Object>( table.getItemIds() ) ) {
				if (!newIds.contains(id))
					table.removeItem(id);
			}
			
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public boolean isShowSearchField() {
		return showSearchField;
	}

	public void setShowSearchField(boolean showSearchField) {
		this.showSearchField = showSearchField;
	}

	public void doUpdateCaptions() {
		table.setCaption(getTableName());
		detailsPanel.setCaption(getDetailsName());
	}
	
	public E getSingleSelected() {
		if (isEditMode())
			return getTarget(editMode);
		Object selectedId = table.getValue();
		if (selectedId == null) return null;
		return getTarget(selectedId);
	}

	public boolean isFullSize() {
		return fullSize;
	}

	public void setFullSize(boolean fullSize) {
		this.fullSize = fullSize;
	}

	@Override
	public MNls getNls() {
		if (nlsBundle == null)
			nlsBundle = MNlsBundle.lookup(this);
		return nlsBundle.getNls(UI.getCurrent().getLocale());
	}

	public boolean isModified() {
		return modified;
	}

	public void setModified(boolean modified) {
		this.modified = modified;
	}

	public boolean isInitialized() {
		return initialized;
	}

	public void setInitialized(boolean initialized) {
		this.initialized = initialized;
	}
	
	@Override
	public synchronized Log log() {
		if (log == null) {
			log = Log.getLog(this);
		}
		return log;
	}

	public boolean isNeedSortUpdate() {
		return needSortUpdate;
	}

	public void setNeedSortUpdate(boolean needSortUpdate) {
		this.needSortUpdate = needSortUpdate;
	}

	public String getSortedColumn() {
		return sortedColumn;
	}

//	public void setSortedColumn(String sortedColumn) {
//		this.sortedColumn = sortedColumn;
//	}

	public boolean isSortedAscending() {
		return sortedAscending;
	}

	public MNlsBundle getNlsBundle() {
		return nlsBundle;
	}

	public void setNlsBundle(MNlsBundle nlsBundle) {
		this.nlsBundle = nlsBundle;
	}

//	public void setSortedAscending(boolean sortedAscending) {
//		this.sortedAscending = sortedAscending;
//	}

}
