package de.mhus.lib.core.logging;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.util.Date;

import de.mhus.lib.core.MCast;
import de.mhus.lib.core.MSingleton;
import de.mhus.lib.core.config.IConfig;
import de.mhus.lib.core.console.Console;
import de.mhus.lib.core.directory.ResourceNode;
import de.mhus.lib.errors.MException;

/**
 * <p>ConsoleFactory class.</p>
 *
 * @author mikehummel
 * @version $Id: $Id
 */
public class ConsoleFactory extends LogFactory {

	//public static boolean tracing = true;
	private PrintStream out;
	private boolean traces = true;
	private boolean printTime = true;
	@SuppressWarnings("unused")
	private ResourceNode config;

	/**
	 * <p>Constructor for ConsoleFactory.</p>
	 */
	public ConsoleFactory() {
		out = System.out;
	}
	
	/** {@inheritDoc} */
	@Override
	public LogEngine createInstance(String name) {
		return new ConsoleLog(name);
	}

	/**
	 * <p>Constructor for ConsoleFactory.</p>
	 *
	 * @param config a {@link de.mhus.lib.core.config.IConfig} object.
	 * @throws java.lang.Exception if any.
	 */
	public ConsoleFactory(IConfig config) throws Exception {
//		name = config.getExtracted("name","");
		init(config);
	}
	
	/**
	 * <p>printTime.</p>
	 *
	 * @return a {@link java.lang.String} object.
	 */
	public String printTime() {
		if (printTime) {
			return MCast.toString(new Date()) + " "; // TODO maybe more efficient
		}
		return "";
	}
	
	/** {@inheritDoc} */
	@Override
	public void init(ResourceNode config) throws Exception {
		if (config == null) return;
		
		this.config = config;
		printTime = config.getBoolean("TIME", printTime);
		String newLevel = config.getExtracted("LEVEL",level.name());
		if (newLevel != null) level = Log.LEVEL.valueOf(newLevel.toUpperCase());
		
		String file = config.getExtracted("file",null);
		String io = config.getExtracted("stream",null);
		traces    = config.getBoolean("traces", true);
		
		if (file != null) {
			FileOutputStream f;
			try {
				f = new FileOutputStream(file,config.getBoolean("append", true));
			} catch (FileNotFoundException e) {
				throw new MException(file,e);
			}
			out = new PrintStream(f,true);
		} else
		if ("console".equals(io)) {
			out = MSingleton.get().getBaseControl().base(this).lookup(Console.class);
		} else
		if ("err".equals(io))
		{
			out = System.err;
		} else {
			out = System.out;
		}
		
	}
	
	/**
	 * <p>Constructor for ConsoleFactory.</p>
	 *
	 * @param stream a {@link java.io.PrintStream} object.
	 */
	public ConsoleFactory(PrintStream stream) {
		this.out = stream;
	}

//	@Override
//    public void update(Observable o, Object arg) {
//        level = Log.LEVEL.valueOf(MSingleton.instance().getConfig().getString("LEVEL",level.name()).toUpperCase()); 
//        super.update(o, arg);
//	}

	/**
	 * <p>getLevel.</p>
	 *
	 * @return a {@link de.mhus.lib.core.logging.Log.LEVEL} object.
	 */
	public Log.LEVEL getLevel() {
		return level;
	}

	/**
	 * <p>setLevel.</p>
	 *
	 * @param level a {@link de.mhus.lib.core.logging.Log.LEVEL} object.
	 */
	public void setLevel(Log.LEVEL level) {
		this.level = level;
	}
	
	private class ConsoleLog extends LogEngine {
		

		public ConsoleLog(String name) {
			super(name);
		}
		
		@Override
		public void debug(Object message) {
			if (!isDebugEnabled()) return;
			out.println(printTime() + "DEBUG: " + getName() + " " + message);
			if (message != null && message instanceof Throwable)
				((Throwable)message).printStackTrace(out);
		}
	
		@Override
		public void debug(Object message, Throwable t) {
			if (!isDebugEnabled()) return;
			out.println(printTime() + "DEBUG: " + getName() + " " + message);
			if (t!=null && traces) t.printStackTrace(out);
		}
	
		@Override
		public void error(Object message) {
			if (!isErrorEnabled()) return;
			out.println(printTime() + "ERROR: " + getName() + " " + message);
			if (message != null && message instanceof Throwable && traces)
				((Throwable)message).printStackTrace(out);
		}
	
		@Override
		public void error(Object message, Throwable t) {
			if (!isErrorEnabled()) return;
			out.println(printTime() + "ERROR: " + getName() + " " + message);
			if (t!=null && traces) t.printStackTrace(out);
		}
	
		@Override
		public void fatal(Object message) {
			if (!isFatalEnabled()) return;
			out.println(printTime() + "FATAL: " + getName() + " " + message);
			if (message != null && message instanceof Throwable && traces)
				((Throwable)message).printStackTrace(out);
		}
	
		@Override
		public void fatal(Object message, Throwable t) {
			if (!isFatalEnabled()) return;
			out.println(printTime() + "FATAL: " + getName() + " " + message);
			if (t!=null && traces) t.printStackTrace(out);
		}
		
		@Override
		public void info(Object message) {
			if (!isInfoEnabled()) return;
			out.println(printTime() + "INFO : " + getName() + " " + message);
			if (message != null && message instanceof Throwable && traces)
				((Throwable)message).printStackTrace(out);
		}
	
		@Override
		public void info(Object message, Throwable t) {
			if (!isInfoEnabled()) return;
			out.println(printTime() + "INFO : " + getName() + " " + message);
			if (t!=null && traces) t.printStackTrace(out);
		}
	
		@Override
		public boolean isDebugEnabled() {
			return getLevel().ordinal() <= Log.LEVEL.DEBUG.ordinal();
		}
	
		@Override
		public boolean isErrorEnabled() {
			return getLevel().ordinal() <= Log.LEVEL.ERROR.ordinal();
		}
	
		@Override
		public boolean isFatalEnabled() {
			return getLevel().ordinal() <= Log.LEVEL.FATAL.ordinal();
		}
	
		@Override
		public boolean isInfoEnabled() {
			return getLevel().ordinal() <= Log.LEVEL.INFO.ordinal();
		}
	
		@Override
		public boolean isTraceEnabled() {
			return getLevel().ordinal() <= Log.LEVEL.TRACE.ordinal();
		}
	
		@Override
		public boolean isWarnEnabled() {
			return getLevel().ordinal() <= Log.LEVEL.WARN.ordinal();
		}
	
		@Override
		public void trace(Object message) {
			if (!isTraceEnabled()) return;
			out.println(printTime() + "TRACE: " + getName() + " " + message);
			if (message != null && message instanceof Throwable && traces)
				((Throwable)message).printStackTrace(out);
		}
	
		@Override
		public void trace(Object message, Throwable t) {
			if (!isTraceEnabled()) return;
			out.println(printTime() + "TRACE: " + getName() + " " + message);
			if (t!=null && traces) t.printStackTrace(out);
		}
	
		@Override
		public void warn(Object message) {
			if (!isWarnEnabled()) return;
			out.println(printTime() + "WARN : " + getName() + " " + message);
			if (message != null && message instanceof Throwable && traces)
				((Throwable)message).printStackTrace(out);
		}
	
		@Override
		public void warn(Object message, Throwable t) {
			if (!isWarnEnabled()) return;
			out.println(printTime() + "WARN : " + getName() + " " + message);
			if (t!=null && traces) t.printStackTrace(out);
		}

		@Override
		public void doInitialize(LogFactory logFactory) {
			
		}

		@Override
		public void close() {
		}
		
	}
	
}
