package de.mhoffrogge.maven.plugins.p2site;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Properties;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

/**
 * This plugin creates template based index.html files for P2 update sites.
 * 
 * @author mhoffrog
 *
 */
@Mojo(name = "build-index-html", defaultPhase = LifecyclePhase.PACKAGE)
public class BuildIndexHtmlMojo extends AbstractMojo {

	/**
	 * The base directory to start processing.
	 */
	@Parameter(defaultValue = "${basedir}", readonly = true, required = true)
	private String baseDir;

	/**
	 * The file to read the updateSite properties from.
	 */
	@Parameter(defaultValue = "updateSite.properties", readonly = true, required = true)
	private String updateSitePropertiesFileName;

	/**
	 * If true, the updateSite properties file will be deleted after processing of that files directory level.
	 */
	@Parameter(defaultValue = "false", readonly = true, required = true)
	private boolean removeUpdateSitePropertiesFile;

	/**
	 * The property name holding the update site name to be shown by the index.html.
	 */
	@Parameter(defaultValue = "update.site.name", readonly = true, required = true)
	private String updateSiteNamePropertyName;

	/**
	 * The property name holding the update site description to be shown by the index.html.
	 */
	@Parameter(defaultValue = "update.site.description", readonly = true, required = true)
	private String updateSiteDescriptionPropertyName;

	/**
	 * The property name holding the update site version to be shown by the index.html.
	 */
	@Parameter(defaultValue = "update.site.version", readonly = true, required = true)
	private String updateSiteVersionPropertyName;

	/**
	 * The property name being replaced for the file listing shown by the index.html.
	 */
	@Parameter(defaultValue = "update.site.contents", readonly = true, required = true)
	private String updateSiteContentsPropertyName;

	/**
	 * Optional: The template file name for an update site URLs index.html.
	 */
	@Parameter(readonly = true, required = false)
	private String indexUpdateSiteTemplateFileName;

	/**
	 * Optional: The template file name for a non update site URLs index.html.
	 */
	@Parameter(readonly = true, required = false)
	private String indexNonUpdateSiteTemplateFileName;

	private String indexUpdateSiteTemplate;

	private String indexNonUpdateSiteTemplate;

	private boolean isIndexHtmlCreated = false;

	private static final String INDEX_HTML_FILE_NAME = "index.html";

	private static final String FOLDER_NAME_IS_VERSION_PATTERN = "^[0-9]+\\.[0-9]+.*";

	public void execute() throws MojoExecutionException, MojoFailureException {
		indexUpdateSiteTemplate = getIndexTemplate(indexUpdateSiteTemplateFileName, "updateSiteTemplate_index.html");
		indexNonUpdateSiteTemplate = getIndexTemplate(indexNonUpdateSiteTemplateFileName,
				"nonUpdateSiteTemplate_index.html");
		getLog().info("Walking directories for " + updateSitePropertiesFileName + " files ...");
		walkAllFiles(new File(baseDir), 0);
		if (!isIndexHtmlCreated) {
			getLog().warn("NO index.html files did have been created.");
		}
	}

	private String getIndexTemplate(final String indexTemplateFileName, final String defaultResourceName)
			throws MojoFailureException {
		InputStream indexTemplateStream = null;
		try {
			if (StringUtils.isNotBlank(indexTemplateFileName)) {
				final File indexTemplateFile = new File(indexTemplateFileName);
				if (!indexTemplateFile.exists()) {
					throw new MojoFailureException(
							"Index template file does not exist: " + indexTemplateFile.getAbsolutePath());
				}
				if (!indexTemplateFile.isFile()) {
					throw new MojoFailureException(
							"Index template file is not a file: " + indexTemplateFile.getAbsolutePath());
				}
				try {
					indexTemplateStream = new FileInputStream(indexTemplateFile);
				} catch (FileNotFoundException e) {
					throw new MojoFailureException("Index template file does not exist or is not readable: "
							+ indexTemplateFile.getAbsolutePath());
				}
			} else {
				indexTemplateStream = getClass().getResourceAsStream(defaultResourceName);
				if (indexTemplateStream == null) {
					throw new MojoFailureException(
							defaultResourceName + " does not exist in this plugins JAR package.");
				}
			}
			try {
				return IOUtils.toString(indexTemplateStream, Charset.defaultCharset());
			} catch (IOException e) {
				throw new MojoFailureException("Failed to read the index template file.", e);
			}
		} finally {
			if (indexTemplateStream != null) {
				try {
					indexTemplateStream.close();
				} catch (IOException e) {
					// nothing to do
				}
			}
		}
	}

	private void walkAllFiles(final File dir, final int depth) throws MojoFailureException {
		if (!dir.exists()) {
			throw new MojoFailureException("Directory does not exist: " + dir.getAbsolutePath());
		}
		if (!dir.isDirectory()) {
			throw new MojoFailureException("This file is not a directory: " + dir.getAbsolutePath());
		}
		final File[] listOfFiles = dir.listFiles();
		if (listOfFiles == null) {
			return;
		}
		// 1. look for updateSiteIndexPropertiesFile
		for (File file : listOfFiles) {
			if (file.isFile() && StringUtils.equalsIgnoreCase(updateSitePropertiesFileName, file.getName())) {
				// we have found it - so we are done for this directory
				getLog().info("Found " + file.getAbsolutePath());
				Properties props = new Properties();
				try (FileInputStream finput = new FileInputStream(file)) {
					props.load(finput);
				} catch (IOException e) {
					throw new MojoFailureException("Failed to read file: " + file.getAbsolutePath(), e);
				}
				createIndexHtml(dir, props, depth);
				if (removeUpdateSitePropertiesFile) {
					try {
						Files.delete(Paths.get(file.toURI()));
						getLog().info("REMOVED " + file.getAbsolutePath());
					} catch (IOException e) {
						throw new MojoFailureException("Failed to remove file: " + file.getAbsolutePath(), e);
					}
				}
				return;
			}
		}
		// 2. look for other directories
		for (File file : listOfFiles) {
			if (file.isDirectory()) {
				walkAllFiles(file, depth + 1);
			}
		}
	}

	private void createIndexHtml(final File dir, Properties props, final int depth) throws MojoFailureException {
		// "clone" properties to preserve values from upper level
		props = new Properties(props);
		// Set version property, if not set and if this folder name matches a version number pattern
		if (StringUtils.isBlank(props.getProperty(updateSiteVersionPropertyName))
				&& dir.getName().matches(FOLDER_NAME_IS_VERSION_PATTERN)) {
			props.setProperty(updateSiteVersionPropertyName, dir.getName());
		}
		Pair<Boolean, String> pairUpdateSiteContent = getUpdateSiteContentFragment(dir, props, depth);
		String indexHtml = pairUpdateSiteContent.getLeft() ? new String(indexUpdateSiteTemplate)
				: new String(indexNonUpdateSiteTemplate);
		indexHtml = replaceIndexHtml(indexHtml, props.getProperty(updateSiteNamePropertyName),
				updateSiteNamePropertyName);
		indexHtml = replaceIndexHtml(indexHtml, props.getProperty(updateSiteDescriptionPropertyName),
				updateSiteDescriptionPropertyName);
		indexHtml = replaceIndexHtml(indexHtml, props.getProperty(updateSiteVersionPropertyName),
				updateSiteVersionPropertyName);
		indexHtml = replaceIndexHtml(indexHtml, pairUpdateSiteContent.getRight(), updateSiteContentsPropertyName);
		final File indexHtmlFile = new File(dir.getAbsolutePath(), INDEX_HTML_FILE_NAME);
		try {
			FileUtils.write(indexHtmlFile, indexHtml, Charset.defaultCharset());
			isIndexHtmlCreated = true;
			getLog().info("  Created " + indexHtmlFile.getAbsolutePath());
		} catch (IOException e) {
			throw new MojoFailureException("Failed to write file: " + indexHtmlFile.getAbsolutePath(), e);
		}
	}

	private static String replaceIndexHtml(final String indexHtml, String value, final String propertyName) {
		if (StringUtils.isBlank(value)) {
			value = StringUtils.EMPTY;
		}
		return StringUtils.replace(indexHtml, "${" + propertyName + "}", value);
	}

	private Pair<Boolean, String> getUpdateSiteContentFragment(final File dir, final Properties props, final int depth)
			throws MojoFailureException {
		boolean isUpdateSite = false;
		StringBuilder sbFolder = new StringBuilder();
		if (depth > 0) {
			sbFolder.append(buildFolderLink(".."));
		}
		final File[] listOfFiles = dir.listFiles();
		if (listOfFiles == null || listOfFiles.length == 0) {
			return Pair.of(isUpdateSite, sbFolder.toString());
		}
		Arrays.sort(listOfFiles);
		StringBuilder sbFiles = new StringBuilder();
		for (File file : listOfFiles) {
			if (file.isFile() && !isExcludedFileName(file.getName())) {
				if (sbFiles.length() > 0) {
					sbFiles.append("\n");
				}
				sbFiles.append(buildFileLink(file.getName()));
				if (!isUpdateSite) {
					isUpdateSite = isUpdateSiteFile(file.getName());
				}
			} else if (file.isDirectory() && !isExcludedDirName(file.getName())) {
				if (sbFolder.length() > 0) {
					sbFolder.append("\n");
				}
				sbFolder.append(buildFolderLink(file.getName()));
				final File probablePropertiesFile = new File(file, updateSitePropertiesFileName);
				if (probablePropertiesFile.exists() && probablePropertiesFile.isFile()) {
					// if there are properties defined on this level, then start a new walk with those
					walkAllFiles(file, depth + 1);
				} else {
					createIndexHtml(file, props, depth + 1);
				}
			}
		}
		sbFolder.append(sbFiles);
		return Pair.of(isUpdateSite, sbFolder.toString());
	}

	private boolean isUpdateSiteFile(String fileName) {
		return StringUtils.equalsIgnoreCase(fileName, "content.jar")
				|| StringUtils.equalsIgnoreCase(fileName, "artifacts.jar")
				|| StringUtils.startsWithIgnoreCase(fileName, "content.xml")
				|| StringUtils.startsWithIgnoreCase(fileName, "artifacts.xml")
				|| StringUtils.equalsIgnoreCase(fileName, "p2.index");
	}

	private boolean isExcludedFileName(String fileName) {
		return StringUtils.equalsIgnoreCase(fileName, updateSitePropertiesFileName)
				|| StringUtils.equalsIgnoreCase(fileName, INDEX_HTML_FILE_NAME)
				|| StringUtils.startsWith(fileName, ".");
	}

	private boolean isExcludedDirName(String dirName) {
		return StringUtils.startsWith(dirName, ".");
	}

	private static String buildFolderLink(String folderName) {
		return String.format(
				"<img src='https://dev.eclipse.org/small_icons/places/folder.png'><a href='%s/'> %s</a><br />",
				folderName, folderName);
	}

	private static String buildFileLink(final String fileName) {
		return String.format(
				"<img src='https://dev.eclipse.org/small_icons/actions/edit-copy.png'><a href='%s'> %s</a><br />",
				fileName, fileName);
	}

}
