/*
 * Decompiled with CFR 0.152.
 */
package de.l3s.icrawl.crawler;

import com.codahale.metrics.Counter;
import com.codahale.metrics.Meter;
import com.codahale.metrics.MetricRegistry;
import com.google.common.collect.Ordering;
import de.l3s.icrawl.crawler.ArchiveCrawlSpecification;
import de.l3s.icrawl.crawler.CrawlUrl;
import de.l3s.icrawl.crawler.CrawledResource;
import de.l3s.icrawl.crawler.analysis.ResourceAnalyser;
import de.l3s.icrawl.crawler.frontier.Frontier;
import de.l3s.icrawl.crawler.io.ArchiveFetcher;
import de.l3s.icrawl.crawler.io.ResultStorer;
import de.l3s.icrawl.crawler.scheduling.StoppingCriterion;
import de.l3s.icrawl.snapshots.Snapshot;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CrawlerThread
implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(CrawlerThread.class);
    private final Frontier frontier;
    private final ArchiveFetcher fetcher;
    private final ResourceAnalyser analyser;
    private final ResultStorer storer;
    private final double relevanceThreshold;
    private final Counter retrieved;
    private final Counter notFound;
    private final Counter irrelevant;
    private final Meter crawlRate;
    private final ArchiveCrawlSpecification spec;
    private boolean stopped = false;
    private final CountDownLatch barrier;
    private final StoppingCriterion stoppingCriterion;

    public CrawlerThread(Frontier frontier, ArchiveFetcher fetcher, ResultStorer storer, ResourceAnalyser analyser, MetricRegistry metrics, ArchiveCrawlSpecification spec, CountDownLatch barrier, StoppingCriterion stoppingCriterion, double relevanceThreshold) {
        this.frontier = frontier;
        this.fetcher = fetcher;
        this.storer = storer;
        this.analyser = analyser;
        this.spec = spec;
        this.barrier = barrier;
        this.stoppingCriterion = stoppingCriterion;
        this.relevanceThreshold = relevanceThreshold;
        this.crawlRate = metrics.meter(MetricRegistry.name(this.getClass(), (String[])new String[]{"crawlRate"}));
        this.retrieved = metrics.counter(MetricRegistry.name(this.getClass(), (String[])new String[]{"retrieved"}));
        this.notFound = metrics.counter(MetricRegistry.name(this.getClass(), (String[])new String[]{"notFound"}));
        this.irrelevant = metrics.counter(MetricRegistry.name(this.getClass(), (String[])new String[]{"irrelevant"}));
    }

    @Override
    public void run() {
        logger.info("Starting crawler thread");
        try {
            while (true) {
                if (Thread.interrupted()) {
                    logger.info("Stopping because of thread interrupt");
                    break;
                }
                if (this.stopped) {
                    logger.info("Stopping because of external stop");
                    break;
                }
                Optional<CrawlUrl> url = this.frontier.pop();
                if (!url.isPresent()) {
                    try {
                        this.stoppingCriterion.updateEmptyQueue();
                        TimeUnit.MILLISECONDS.sleep(10L);
                        continue;
                    }
                    catch (InterruptedException e) {
                        logger.info("Interrupted while waiting for URLs to arrive, stopping");
                        break;
                    }
                }
                this.fetch(url.get());
            }
            this.fetcher.close();
        }
        catch (Throwable t) {
            logger.info("Very unexpected exception", t);
        }
        finally {
            logger.info("Crawler thread finished");
            this.barrier.countDown();
        }
    }

    private void fetch(CrawlUrl crawlUrl) {
        try {
            this.crawlRate.mark();
            List<Snapshot> snapshots = this.fetcher.get(crawlUrl, this.spec.getReferenceTime());
            if (snapshots.isEmpty()) {
                this.storer.storeNotFound(crawlUrl);
                this.notFound.inc();
                this.stoppingCriterion.updateFailure();
            } else {
                ResourceAnalyser.Result bestResult = null;
                Snapshot bestSnapshot = null;
                double minRelevance = Double.POSITIVE_INFINITY;
                double maxRelevance = Double.NEGATIVE_INFINITY;
                ZonedDateTime earliestDate = ZonedDateTime.now();
                ZonedDateTime latestDate = ZonedDateTime.ofInstant(Instant.ofEpochMilli(0L), ZoneOffset.UTC);
                for (Snapshot snapshot : snapshots) {
                    ResourceAnalyser.Result result = this.analyser.analyse(snapshot, crawlUrl);
                    double relevance = result.getRelevance();
                    if (bestResult == null || bestResult.getRelevance() < relevance) {
                        bestResult = result;
                        bestSnapshot = snapshot;
                    }
                    if (relevance >= 0.0) {
                        minRelevance = Double.min(minRelevance, relevance);
                        maxRelevance = Double.max(maxRelevance, relevance);
                    }
                    if (result.getModifiedDate() == null) continue;
                    earliestDate = (ZonedDateTime)Ordering.natural().min((Object)earliestDate, (Object)result.getModifiedDate());
                    latestDate = (ZonedDateTime)Ordering.natural().max((Object)latestDate, (Object)result.getModifiedDate());
                }
                if (this.stopped) {
                    return;
                }
                assert (bestResult != null);
                if (bestResult.getRelevance() < this.relevanceThreshold) {
                    this.irrelevant.inc();
                    this.stoppingCriterion.updateIrrelevant(bestResult.getRelevance());
                } else {
                    this.frontier.push(bestResult.getOutlinks());
                    this.storer.store(new CrawledResource(crawlUrl, bestSnapshot, bestResult.getRelevance(), bestResult.getModifiedDate(), Duration.between(earliestDate, latestDate), minRelevance, maxRelevance));
                    this.retrieved.inc();
                    this.stoppingCriterion.updateSuccess(bestResult.getRelevance());
                }
            }
        }
        catch (IOException e) {
            logger.info("Exception while fetching '{}', skipping ", (Object)crawlUrl, (Object)e);
        }
        catch (Exception e) {
            logger.info("Unexpected exception ", (Throwable)e);
        }
    }

    public void stop() {
        this.stopped = true;
    }
}

