/**
 * This software is licensed under the MIT license. 
 */
package de.kune.sessionxs.matcher;

import javax.servlet.http.HttpServletRequest;

import de.kune.sessionxs.SessionMatcherFilter;
import de.kune.sessionxs.matcher.context.SessionMatcherContext;

/**
 * Specifies the interface for a session matcher. Session matchers will be
 * called by the {@link SessionMatcherFilter} to match a
 * {@link SessionMatcherContext} against a request. Please be aware that session
 * matchers will be shared between different requests and sessions, so any state
 * information should be stored into the session matcher context, only. If
 * possible implementations of this interface should inherit from
 * {@link AbstractSessionMatcher} to ensure compatibility with future releases.
 * 
 * @author Alexander Kune
 * 
 */
public interface SessionMatcher {

	/**
	 * Initializes the session matcher from a request. This is the initial call
	 * to a session matcher. Its purpose is to determine the parameters against
	 * which subsequent requests will be checked. These parameters have to be
	 * stored into the session matcher context returned by this method.
	 * 
	 * @param request
	 *            the request
	 * @return a session matcher context
	 */
	SessionMatcherContext init(HttpServletRequest request);

	/**
	 * Determines if a request actually meets the criteria represented by the
	 * specified session matcher context. The session matcher context is the on
	 * created by the {@link #init(HttpServletRequest)} method.
	 * 
	 * @param context
	 *            the session matcher context
	 * @param request
	 *            the request
	 * @return true if the request meets the criteria represented by the session
	 *         matcher context
	 */
	boolean matches(SessionMatcherContext context, HttpServletRequest request);

}
