/**
 * This software is licensed under the MIT license. 
 */
package de.kune.sessionxs.matcher;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.kune.sessionxs.matcher.context.SessionMatcherContext;
import de.kune.sessionxs.matcher.context.StringValueContext;

/**
 * An abstract header matcher that can be used by subclasses for simple header
 * value-based matching.
 * 
 * @author Alexander Kune
 * 
 */
public abstract class AbstractHeaderMatcher extends AbstractSessionMatcher
		implements SessionMatcher {

	private static final Logger LOGGER = LoggerFactory
			.getLogger(AbstractHeaderMatcher.class);

	@Override
	public SessionMatcherContext init(HttpServletRequest request) {
		return new StringValueContext(request.getHeader(getHeaderKey()));
	}

	@Override
	public boolean matches(SessionMatcherContext context,
			HttpServletRequest request) {
		if (context instanceof StringValueContext) {
			if (LOGGER.isDebugEnabled()) {
				LOGGER.debug("Context value: ["
						+ ((StringValueContext) context).getValue()
						+ "], request value: ["
						+ request.getHeader(getHeaderKey()) + "]");
			}
			return ((StringValueContext) context).getValue().equals(
					request.getHeader(getHeaderKey()));
		} else {
			throw new IllegalArgumentException("Context must be of class ["
					+ StringValueContext.class.getName() + "].");
		}
	}

	/**
	 * Specifies the header key to use for header-based matching.
	 * 
	 * @return the header key
	 */
	protected abstract String getHeaderKey();

}
