/*
 * Decompiled with CFR 0.152.
 */
package me.bazhenov.docker;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Scanner;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import me.bazhenov.docker.ContainerDefinition;
import me.bazhenov.docker.VolumeDef;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testng.util.Strings;

public final class Docker
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(Docker.class);
    private static final ObjectMapper jsonReader = new ObjectMapper();
    List<String> tcpFiles = Arrays.asList("/proc/self/net/tcp", "/proc/self/net/tcp6");
    private final String pathToDocker;
    private final Set<String> containersToRemove = ConcurrentHashMap.newKeySet();
    private final Set<String> networks = ConcurrentHashMap.newKeySet();

    public Docker(String pathToDocker) {
        this.pathToDocker = Objects.requireNonNull(pathToDocker);
    }

    public Docker() {
        this("docker");
    }

    public String executeAndReturnOutput(ContainerDefinition definition) throws IOException, InterruptedException {
        List<String> cmd = this.prepareDockerCommand(definition, new String[0]);
        return Docker.doExecuteAndGetFullOutput(cmd);
    }

    public String start(ContainerDefinition definition) throws IOException, InterruptedException {
        this.ensureImageAvailable(definition.getImage());
        this.createNetwork(definition.getNetwork());
        File cidFile = File.createTempFile("docker", "cid");
        cidFile.deleteOnExit();
        if (!cidFile.delete()) {
            throw new IllegalStateException("Docker requires cid-file to be not present at the moment of starting a container");
        }
        List<String> cmd = this.prepareDockerCommand(definition, "--cidfile", cidFile.getAbsolutePath());
        Process process = Docker.runProcess(cmd);
        try {
            String cid = this.waitForCid(process, cidFile);
            if (definition.isRemoveAfterCompletion()) {
                this.containersToRemove.add(cid);
            }
            this.waitForContainerRun(cid, process);
            if (Docker.shouldWaitForOpenPorts(definition)) {
                this.waitForPorts(cid, definition.getPublishedPorts().keySet());
            }
            return cid;
        }
        catch (IOException e) {
            throw new UncheckedIOException("Unable to start container\nContainer stderr: " + Docker.readFully(process.getErrorStream()), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void createNetwork(String network) throws IOException, InterruptedException {
        Set<String> set = this.networks;
        synchronized (set) {
            if (!Strings.isNullOrEmpty((String)network) && this.networks.add(network)) {
                this.docker("network", "create", network);
            }
        }
    }

    private void waitForContainerRun(String cid, Process process) throws IOException, InterruptedException {
        String state;
        while (true) {
            if ("running".equalsIgnoreCase(state = this.getContainerState(cid))) {
                return;
            }
            if (!"created".equalsIgnoreCase(state)) break;
            Thread.sleep(100L);
        }
        if (!process.isAlive() && process.exitValue() != 0) {
            throw new IllegalStateException("Unable to start container " + cid + "\nExit code: " + process.exitValue() + "\nStderr: " + Docker.readFully(process.getErrorStream()));
        }
        throw new IllegalStateException("Unable to start container " + cid + " current state is " + state);
    }

    private static boolean shouldWaitForOpenPorts(ContainerDefinition definition) {
        return !definition.getPublishedPorts().isEmpty() && definition.isWaitForAllExposedPortsToBeOpen();
    }

    private void ensureImageAvailable(String image) throws IOException, InterruptedException {
        int exitValue = Docker.doExecute(Arrays.asList(new String[]{this.pathToDocker, "image", "inspect", image}), new HashSet<Integer>(Arrays.asList(new Integer[]{Integer.valueOf((int)0), Integer.valueOf((int)1)}))).exitCode;
        if (exitValue == 1) {
            log.warn("Image {} is not found locally. It will take some time to download it.", (Object)image);
        }
    }

    private String waitForCid(Process process, File cidFile) throws InterruptedException, IOException {
        while (!cidFile.isFile() || cidFile.length() <= 0L) {
            if (!process.isAlive() && process.exitValue() != 0) {
                throw new IllegalStateException("Unable to start Docker container.\nExit code: " + process.exitValue() + "\nStderr: " + Docker.readFully(process.getErrorStream()));
            }
            Thread.sleep(100L);
        }
        return Files.readAllLines(cidFile.toPath()).get(0);
    }

    private static String doExecuteAndGetFullOutput(List<String> cmd) throws IOException, InterruptedException {
        return Docker.doExecute(cmd, Collections.singleton(Integer.valueOf((int)0))).standardOutput;
    }

    private static ExecutionResult doExecute(List<String> cmd, Set<Integer> expectedExitCodes) throws IOException, InterruptedException {
        Process process = Docker.runProcess(cmd);
        String processOutput = Docker.readFully(process.getInputStream());
        String processError = Docker.readFully(process.getErrorStream());
        int exitCode = process.waitFor();
        if (!expectedExitCodes.contains(exitCode)) {
            throw new IOException("Unable to execute: " + String.join((CharSequence)" ", cmd) + "\nExit code: " + exitCode + "\nStderr: " + processError + "\nStdout: " + processOutput);
        }
        return new ExecutionResult(exitCode, processOutput, processError);
    }

    private static Process runProcess(List<String> cmd) throws IOException {
        if (log.isDebugEnabled()) {
            log.debug("Executing: {}", (Object)Docker.prettyFormatCommand(cmd));
        }
        ProcessBuilder builder = new ProcessBuilder(cmd);
        return builder.start();
    }

    private List<String> prepareDockerCommand(ContainerDefinition def, String ... additionalOpts) {
        String string;
        String network;
        ArrayList<String> cmd = new ArrayList<String>();
        cmd.add(this.pathToDocker);
        cmd.add("run");
        cmd.add("-l");
        cmd.add("docker");
        if (additionalOpts.length > 0) {
            cmd.addAll(Arrays.asList(additionalOpts));
        }
        def.getPublishedPorts().forEach((key, value) -> {
            cmd.add("-p");
            cmd.add(value > 0 ? value + ":" + key : String.valueOf(key));
        });
        for (VolumeDef volumeDef : def.getVolumes()) {
            File location = volumeDef.getLocation();
            String mountPoint = volumeDef.getMountPoint();
            if (location == null) {
                cmd.add("-v");
                cmd.add(mountPoint);
                continue;
            }
            Docker.ensureVolumeCanBeMounted(volumeDef, location);
            cmd.add("-v");
            cmd.add(location.getAbsolutePath() + ":" + mountPoint);
        }
        for (Map.Entry entry : def.getEnvironment().entrySet()) {
            cmd.add("-e");
            cmd.add((String)entry.getKey() + "=" + (String)entry.getValue());
        }
        if (def.isRemoveAfterCompletion()) {
            cmd.add("--rm");
        }
        if (def.getWorkingDirectory() != null) {
            cmd.add("-w");
            cmd.add(def.getWorkingDirectory());
        }
        if (!Strings.isNullOrEmpty((String)(network = def.getNetwork()))) {
            cmd.add("--network=" + network);
        }
        if (!Strings.isNullOrEmpty((String)(string = def.getNetworkAlias()))) {
            cmd.add("--network-alias=" + string);
        }
        cmd.addAll(def.getCustomOptions());
        cmd.add(def.getImage());
        cmd.addAll(def.getCommand());
        return cmd;
    }

    private static void ensureVolumeCanBeMounted(VolumeDef volume, File location) {
        if (!location.exists()) {
            if (volume.isCreateDirectoryIfMissing()) {
                if (!location.mkdirs()) {
                    throw new IllegalStateException("Unable to create directory: " + location);
                }
            } else {
                throw new IllegalStateException("No file directory at: " + location);
            }
        }
    }

    private static String prettyFormatCommand(List<String> cmd) {
        return cmd.stream().map(c -> c.contains(" ") ? "'" + c + "'" : c).collect(Collectors.joining(" "));
    }

    static String readFully(InputStream stream) {
        Scanner scanner = new Scanner(stream).useDelimiter("\\A");
        return scanner.hasNext() ? scanner.next() : "";
    }

    private void waitForPorts(String cid, Set<Integer> ports) throws IOException, InterruptedException {
        Thread self = Thread.currentThread();
        long start = System.currentTimeMillis();
        boolean reported = false;
        while (!self.isInterrupted()) {
            HashSet<Integer> openPorts = new HashSet<Integer>();
            for (String file : this.tcpFiles) {
                openPorts.addAll(Docker.readListenPorts(this.docker("exec", cid, "sh", "-c", "[ ! -f " + file + " ] || cat " + file)));
            }
            if (openPorts.containsAll(ports)) {
                return;
            }
            this.checkContainerRunning(cid);
            if (!reported && System.currentTimeMillis() - start > 5000L) {
                reported = true;
                log.warn("Waiting for ports {} to open in container {}", ports, (Object)cid);
            }
            TimeUnit.MILLISECONDS.sleep(200L);
        }
    }

    private String docker(String command, String ... args) throws IOException, InterruptedException {
        ArrayList<String> cmd = new ArrayList<String>(args.length + 2);
        cmd.add(this.pathToDocker);
        cmd.add(command);
        cmd.addAll(Arrays.asList(args));
        return Docker.doExecuteAndGetFullOutput(cmd);
    }

    private void checkContainerRunning(String id) throws IOException, InterruptedException {
        String state = this.getContainerState(id);
        if (!"running".equalsIgnoreCase(state)) {
            throw new IllegalStateException("Container " + id + " failed to start. Current state: " + state);
        }
    }

    private String getContainerState(String id) throws IOException, InterruptedException {
        String json = this.docker("inspect", id);
        JsonNode root = jsonReader.readTree(json);
        return root.at("/0/State/Status").asText();
    }

    public Map<Integer, Integer> getPublishedTcpPorts(String containerName) throws IOException, InterruptedException {
        String json = this.docker("inspect", containerName);
        JsonNode root = jsonReader.readTree(json);
        return Docker.doGetPublishedPorts(root);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        if (!this.containersToRemove.isEmpty()) {
            try {
                ArrayList<String> cmd = new ArrayList<String>(Arrays.asList(this.pathToDocker, "rm", "-f", "-v"));
                cmd.addAll(this.containersToRemove);
                Docker.doExecute(cmd, Collections.singleton(0));
                this.containersToRemove.clear();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        Set<String> set = this.networks;
        synchronized (set) {
            if (!this.networks.isEmpty()) {
                try {
                    for (String network : this.networks) {
                        this.docker("network", "rm", network);
                    }
                    this.networks.clear();
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    static Map<Integer, Integer> doGetPublishedPorts(JsonNode root) {
        JsonNode candidate = root.at("/0/NetworkSettings/Ports");
        if (candidate.isMissingNode() || candidate.isNull()) {
            return Collections.emptyMap();
        }
        ObjectNode ports = (ObjectNode)candidate;
        Iterator names = ports.fieldNames();
        HashMap<Integer, Integer> pts = new HashMap<Integer, Integer>();
        while (names.hasNext()) {
            String field = (String)names.next();
            if (!field.matches("\\d+/tcp")) continue;
            String[] parts = field.split("/", 2);
            int containerPort = Integer.parseInt(parts[0]);
            int localPort = ports.at("/" + field.replace("/", "~1") + "/0/HostPort").asInt();
            pts.put(containerPort, localPort);
        }
        return pts;
    }

    public static Set<Integer> readListenPorts(String output) {
        Scanner scanner = new Scanner(output);
        scanner.useRadix(16).useDelimiter("[\\s:]+");
        HashSet<Integer> result = new HashSet<Integer>();
        if (scanner.hasNextLine()) {
            scanner.nextLine();
        }
        while (scanner.hasNextLine()) {
            scanner.nextInt();
            scanner.next();
            int localPort = scanner.nextInt();
            result.add(localPort);
            scanner.nextLine();
        }
        return result;
    }

    int getVolumesCount() throws IOException, InterruptedException {
        ArrayList<String> cmd = new ArrayList<String>();
        cmd.add(this.pathToDocker);
        cmd.add("volume");
        cmd.add("ls");
        cmd.add("-q");
        String out = Docker.doExecuteAndGetFullOutput(cmd);
        String[] parts = out.trim().split("\n");
        return parts.length;
    }

    private static class ExecutionResult {
        int exitCode;
        String standardOutput;
        String errorOutput;

        public ExecutionResult(int exitCode, String processOutput, String processError) {
            this.exitCode = exitCode;
            this.standardOutput = processOutput;
            this.errorOutput = processError;
        }
    }
}

