/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package de.knightsoftnet.gwtp.spring.shared.search;

import de.knightsoftnet.validators.shared.data.FieldTypeEnum;

import java.util.Arrays;
import java.util.Collection;
import java.util.Objects;

public class SearchFieldDefinition {
  private final String fieldName;
  private final String fieldDisplayName;
  private final FieldTypeEnum fieldType;
  private final Collection<SearchOperation> allowedSearchOperations;

  /**
   * constructor initializing all fields.
   */
  public SearchFieldDefinition(final String fieldName, final String fieldDisplayName,
      final FieldTypeEnum fieldType) {
    this(fieldName, fieldDisplayName, fieldType,
        SearchFieldDefinition.defaultSearchOptions(fieldType));
  }

  /**
   * constructor initializing all fields.
   */
  public SearchFieldDefinition(final String fieldName, final String fieldDisplayName,
      final FieldTypeEnum fieldType, final SearchOperation... allowedSearchOperations) {
    super();
    this.fieldName = fieldName;
    this.fieldDisplayName = fieldDisplayName;
    this.fieldType = fieldType;
    this.allowedSearchOperations = Arrays.asList(allowedSearchOperations);
  }

  private static SearchOperation[] defaultSearchOptions(final FieldTypeEnum fieldType) {
    switch (fieldType) {
      case BOOLEAN:
        return new SearchOperation[] {SearchOperation.EQUALITY, //
            SearchOperation.NEGATION};
      case DATE:
      case TIME:
      case DATETIME:
        return new SearchOperation[] {SearchOperation.EQUALITY, //
            SearchOperation.NEGATION, //
            SearchOperation.GREATER_THEN, //
            SearchOperation.GREATER_OR_EQUAL_THEN, //
            SearchOperation.LESS_THEN, //
            SearchOperation.LESS_OR_EQUAL_THEN};
      default:
        return SearchOperation.values();
    }
  }

  public String getFieldName() {
    return fieldName;
  }

  public String getFieldDisplayName() {
    return fieldDisplayName;
  }

  public FieldTypeEnum getFieldType() {
    return fieldType;
  }

  public Collection<SearchOperation> getAllowedSearchOperations() {
    return allowedSearchOperations;
  }

  @Override
  public int hashCode() {
    return Objects.hashCode(fieldName);
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    final SearchFieldDefinition other = (SearchFieldDefinition) obj;
    return Objects.equals(fieldName, other.fieldName);
  }

  @Override
  public String toString() {
    return fieldName;
  }
}
