/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package de.knightsoftnet.mtwidgets.client.ui.widget;

import de.knightsoftnet.mtwidgets.client.ui.widget.features.HasAutocomplete;
import de.knightsoftnet.mtwidgets.client.ui.widget.features.HasAutofocus;
import de.knightsoftnet.mtwidgets.client.ui.widget.features.HasDataList;
import de.knightsoftnet.mtwidgets.client.ui.widget.features.HasFormNoValidate;
import de.knightsoftnet.mtwidgets.client.ui.widget.features.HasPlaceholder;
import de.knightsoftnet.mtwidgets.client.ui.widget.features.HasRequired;
import de.knightsoftnet.mtwidgets.client.ui.widget.features.HasValidationMessageElement;
import de.knightsoftnet.mtwidgets.client.ui.widget.features.HasValidationPattern;
import de.knightsoftnet.mtwidgets.client.ui.widget.features.HasValidity;
import de.knightsoftnet.mtwidgets.client.ui.widget.helper.ElementCast;
import de.knightsoftnet.mtwidgets.client.ui.widget.helper.ErrorMessageFormater;
import de.knightsoftnet.mtwidgets.client.ui.widget.helper.FeatureCheck;

import com.google.gwt.dom.client.Element;
import com.google.gwt.editor.client.EditorError;
import com.google.gwt.editor.client.HasEditorErrors;
import com.google.gwt.text.shared.Parser;
import com.google.gwt.text.shared.Renderer;
import com.google.gwt.user.client.ui.HTMLPanel;
import com.google.gwt.user.client.ui.ValueBoxBase;

import elemental2.dom.HTMLInputElement;
import elemental2.dom.ValidityState;

import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public class ValueBoxBaseWithEditorErrors<T> extends ValueBoxBase<T> implements HasEditorErrors<T>,
    HasValidationMessageElement, HasAutofocus, HasRequired, HasValidity, HasPlaceholder,
    HasValidationPattern, HasFormNoValidate, HasDataList, HasAutocomplete {

  private HTMLPanel validationMessageElement;

  protected ValueBoxBaseWithEditorErrors(final Element elem, final Renderer<T> renderer,
      final Parser<T> parser) {
    super(elem, renderer, parser);
  }

  public ValueBoxBaseWithEditorErrors(final HTMLInputElement element, final Renderer<T> renderer,
      final Parser<T> parser) {
    super(ElementCast.cast(element), renderer, parser);
  }

  @Override
  public void showErrors(final List<EditorError> perrors) {
    final Set<String> messages = perrors.stream().filter(error -> this.editorErrorMatches(error))
        .map(error -> error.getMessage()).collect(Collectors.toSet());
    this.showErrors(messages);
  }

  /**
   * show error messages.
   *
   * @param messages set of messages
   */
  public void showErrors(final Set<String> messages) {
    final HTMLInputElement inputElement = this.getInputElement();
    if (messages == null || messages.isEmpty()) {
      if (FeatureCheck.supportCustomValidity(inputElement)) {
        inputElement.setCustomValidity(StringUtils.EMPTY);
      }
      if (this.validationMessageElement == null) {
        inputElement.title = StringUtils.EMPTY;
      } else {
        this.validationMessageElement.getElement().removeAllChildren();
      }
    } else {
      final String messagesAsString = ErrorMessageFormater.messagesToString(messages);
      if (FeatureCheck.supportCustomValidity(inputElement)) {
        inputElement.setCustomValidity(messagesAsString);
      }
      if (this.validationMessageElement == null) {
        inputElement.title = messagesAsString;
      } else {
        this.validationMessageElement.getElement()
            .setInnerSafeHtml(ErrorMessageFormater.messagesToList(messages));
      }
    }
  }

  /**
   * Checks if a error belongs to this widget.
   *
   * @param perror editor error to check
   * @return true if the error belongs to this widget
   */
  protected boolean editorErrorMatches(final EditorError perror) {
    return perror != null && perror.getEditor() != null
        && (equals(perror.getEditor()) || perror.getEditor().equals(asEditor()));
  }

  public HTMLInputElement getInputElement() {
    return (HTMLInputElement) ElementCast.cast(getElement());
  }

  @Override
  public String getValidationMessage() {
    return this.getInputElement().validationMessage;
  }

  @Override
  public ValidityState getValidity() {
    return this.getInputElement().validity;
  }

  @Override
  public boolean checkValidity() {
    return this.getInputElement().checkValidity();
  }

  @Override
  public boolean isFormNoValidate() {
    return this.getInputElement().formNoValidate;
  }

  @Override
  public void setFormNoValidate(final boolean arg) {
    this.getInputElement().formNoValidate = arg;
  }

  @Override
  public boolean isRequired() {
    return this.getInputElement().required;
  }

  @Override
  public void setRequired(final boolean arg) {
    this.getInputElement().required = arg;
  }

  @Override
  public String getPattern() {
    return this.getInputElement().pattern;
  }

  @Override
  public void setPattern(final String arg) {
    this.getInputElement().pattern = arg;
  }


  @Override
  public String getPlaceholder() {
    return this.getInputElement().placeholder;
  }

  @Override
  public void setPlaceholder(final String arg) {
    this.getInputElement().placeholder = arg;
  }

  @Override
  public boolean isAutofocus() {
    return this.getInputElement().autofocus;
  }

  @Override
  public void setAutofocus(final boolean arg) {
    this.getInputElement().autofocus = arg;
  }

  @Override
  public void setValidationMessageElement(final HTMLPanel pelement) {
    this.validationMessageElement = pelement;
  }

  @Override
  public HTMLPanel getValidationMessageElement() {
    return this.validationMessageElement;
  }

  @Override
  public void setDataListWidget(final DataListWidget pdataListWidget) {
    this.getInputElement().setAttribute("list", pdataListWidget.getElement().getId());
  }

  @Override
  public String getAutocomplete() {
    return this.getInputElement().autocomplete;
  }

  @Override
  public void setAutocomplete(final String arg) {
    this.getInputElement().autocomplete = arg;
  }
}
