/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package de.knightsoftnet.mtwidgets.client.ui.widget;

import de.knightsoftnet.mtwidgets.client.jswrapper.JQuery;
import de.knightsoftnet.mtwidgets.client.ui.handler.HandlerFactory;
import de.knightsoftnet.mtwidgets.client.ui.widget.helper.DateParser;
import de.knightsoftnet.mtwidgets.client.ui.widget.helper.DateRenderer;
import de.knightsoftnet.mtwidgets.client.ui.widget.helper.DateTimeLocalParser;
import de.knightsoftnet.mtwidgets.client.ui.widget.helper.DateTimeLocalRenderer;
import de.knightsoftnet.mtwidgets.client.ui.widget.helper.TimeParser;
import de.knightsoftnet.mtwidgets.client.ui.widget.helper.TimeRenderer;
import de.knightsoftnet.mtwidgets.client.ui.widget.resourceloader.WebshimResources;
import de.knightsoftnet.validators.shared.data.FieldTypeEnum;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.text.client.DoubleParser;
import com.google.gwt.text.client.DoubleRenderer;
import com.google.gwt.text.shared.testing.PassthroughParser;
import com.google.gwt.text.shared.testing.PassthroughRenderer;

import elemental2.dom.CSSProperties;
import elemental2.dom.DomGlobal;
import elemental2.dom.HTMLInputElement;
import elemental2.dom.ValidityState;

import org.apache.commons.lang3.StringUtils;

import java.text.ParseException;

/**
 * Dynamic input widget which can switch between the input types.
 *
 * @author Manfred Tremmel
 */
public class DynamicInputWidget extends ValueBox<String> {

  private static final String TYPE = "type";
  private FieldTypeEnum fieldType;
  private boolean nativeSupport;
  private HandlerRegistration attachedHandler;

  /**
   * default constructor.
   */
  public DynamicInputWidget() {
    super((HTMLInputElement) DomGlobal.document.createElement("input"), "text",
        PassthroughRenderer.instance(), PassthroughParser.instance());
    fieldType = FieldTypeEnum.STRING;
  }

  public final FieldTypeEnum getFieldType() {
    return fieldType;
  }

  /**
   * set type of the field.
   *
   * @param pfieldType FieldTypeEnum
   * @param prequired true if field is required
   */
  public final void setFieldType(final FieldTypeEnum pfieldType, final boolean prequired) {
    if (attachedHandler != null) {
      attachedHandler.removeHandler();
      attachedHandler = null;
    }
    fieldType = pfieldType;
    final HTMLInputElement inputElement = getInputElement();
    inputElement.required = prequired;
    inputElement.style.width = CSSProperties.WidthUnionType.of(null);
    switch (pfieldType) {
      case NUMERIC:
        inputElement.setAttribute(DynamicInputWidget.TYPE, "text");
        attachedHandler = addKeyPressHandler(HandlerFactory.getDecimalKeyPressHandler());
        break;
      case BOOLEAN:
        inputElement.setAttribute(DynamicInputWidget.TYPE, "checkbox");
        inputElement.style.width = CSSProperties.WidthUnionType.of("auto");
        break;
      case DATE:
        inputElement.setAttribute(DynamicInputWidget.TYPE, "date");
        checkNative();
        break;
      case TIME:
        inputElement.setAttribute(DynamicInputWidget.TYPE, "time");
        checkNative();
        break;
      case DATETIME:
        inputElement.setAttribute(DynamicInputWidget.TYPE, "datetime-local");
        checkNative();
        break;
      default:
        inputElement.setAttribute(DynamicInputWidget.TYPE, "text");
        break;
    }
  }

  @Override
  public String getValueOrThrow() throws ParseException {
    final HTMLInputElement inputElement = getInputElement();
    final ValidityState validityState = inputElement.validity;
    if (validityState.typeMismatch) {
      throw new ParseException("invalid input", 0);
    }
    final String valueAsString = inputElement.value;
    switch (fieldType) {
      case NUMERIC:
        if (StringUtils.isEmpty(valueAsString)) {
          return null;
        }
        return Double.toString(DoubleParser.instance().parse(valueAsString));
      case BOOLEAN:
        return Boolean.toString(inputElement.checked);
      case DATE:
        if (StringUtils.isEmpty(valueAsString)) {
          return null;
        }
        return DateRenderer.instance().render(DateParser.instance().parse(valueAsString));
      case TIME:
        if (StringUtils.isEmpty(valueAsString)) {
          return null;
        }
        return TimeRenderer.instance().render(TimeParser.instance().parse(valueAsString));
      case DATETIME:
        if (StringUtils.isEmpty(valueAsString)) {
          return null;
        }
        return DateTimeLocalRenderer.instance()
            .render(DateTimeLocalParser.instance().parse(valueAsString));
      default:
        return StringUtils.trimToNull(valueAsString);
    }
  }

  @Override
  public void setValue(final String pvalue, final boolean pfireEvents) {
    final String oldValue = getValue();
    final HTMLInputElement inputElement = getInputElement();
    switch (fieldType) {
      case NUMERIC:
        if (StringUtils.isEmpty(pvalue)) {
          inputElement.value = pvalue;
        } else {
          inputElement.value = DoubleRenderer.instance().render(Double.parseDouble(pvalue));
        }
        break;
      case BOOLEAN:
        inputElement.checked = StringUtils.equalsIgnoreCase(pvalue, "true");
        break;
      case DATE:
      case TIME:
      case DATETIME:
        inputElement.value = pvalue;
        if (!nativeSupport && WebshimResources.isInitialized()) {
          getJQueryElement().val(pvalue);
        }
        break;
      default:
        inputElement.value = pvalue;
        break;
    }
    if (pfireEvents) {
      ValueChangeEvent.fireIfNotEqual(this, oldValue, pvalue);
    }
  }

  private void checkNative() {
    final String rememberValue = getInputElement().value;
    getInputElement().value = "x";
    final String dateInput = getInputElement().value;
    getInputElement().value = rememberValue;
    nativeSupport = !StringUtils.equals(dateInput, "x");
    if (!nativeSupport) {
      WebshimResources.whenReady(event -> {
        Scheduler.get().scheduleFixedDelay(() -> {
          getJQueryElement().updatePolyfill();
          GWT.log("Initialize webshim for: " + getElement().getId());
          try {
            getJQueryElement().getShadowElement().addEventListener("change",
                listener -> ValueChangeEvent.fire(this, getValue()));
          } catch (final Exception exception) {
            GWT.log(exception.getMessage());
          }
          return false;
        }, 1000);
      });
    }
  }

  private JQuery getJQueryElement() {
    return JQuery.$("#" + getElement().getId());
  }
}
