/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package de.knightsoftnet.validators.client.editor.impl;

import de.knightsoftnet.validators.client.decorators.ExtendedValueBoxEditor;
import de.knightsoftnet.validators.client.decorators.IsDecorator;
import de.knightsoftnet.validators.client.editor.CheckTimeEnum;

import com.google.gwt.event.dom.client.HasKeyPressHandlers;
import com.google.gwt.event.dom.client.HasKeyUpHandlers;
import com.google.gwt.event.dom.client.KeyPressHandler;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.logical.shared.HasValueChangeHandlers;
import com.google.gwt.event.logical.shared.ValueChangeHandler;

import org.gwtproject.editor.client.EditorContext;
import org.gwtproject.editor.client.impl.Initializer;

/**
 * Bean validation initializer.
 *
 * @author Manfred Tremmel
 */
public class BeanValidationInitializer extends Initializer {

  /**
   * handler which commits when return is pressed.
   */
  private final KeyPressHandler commitOnReturnHandler;

  /**
   * handler which starts validates on key up.
   */
  private final KeyUpHandler validateOnKeyUpHandler;

  /**
   * handler which starts validation on value changes.
   */
  private final ValueChangeHandler<?> validateOnVueChangeHandler;

  /**
   * handler which handles value changes.
   */
  private final ValueChangeHandler<?> valueChangeHandler;

  /**
   * check input time.
   */
  private final CheckTimeEnum checkTime;

  /**
   * constructor.
   *
   * @param commitOnReturnHandler commit on return handler
   * @param validateOnKeyUpHandler on key up handler
   * @param validateOnVueChangeHandler on vue change handler
   * @param valueChangeHandler value change handler
   * @param checkTime enumeration which describes when to do the check
   * @param submitOnReturn submit form if return is pressed in a input field
   */
  public BeanValidationInitializer(final KeyPressHandler commitOnReturnHandler,
      final KeyUpHandler validateOnKeyUpHandler,
      final ValueChangeHandler<?> validateOnVueChangeHandler,
      final ValueChangeHandler<?> valueChangeHandler, final CheckTimeEnum checkTime,
      final boolean submitOnReturn) {
    super();
    this.commitOnReturnHandler = submitOnReturn ? commitOnReturnHandler : null;
    this.validateOnKeyUpHandler =
        checkTime == CheckTimeEnum.ON_KEY_UP ? validateOnKeyUpHandler : null;
    this.validateOnVueChangeHandler =
        checkTime == CheckTimeEnum.ON_CHANGE || checkTime == CheckTimeEnum.ON_KEY_UP
            ? validateOnVueChangeHandler
            : null;
    this.valueChangeHandler = valueChangeHandler;
    this.checkTime = checkTime;
  }

  @Override
  public <Q> boolean visit(final EditorContext<Q> ctx) {
    Object editor = ctx.getEditor();
    final boolean result = super.visit(ctx);
    if (ctx.getEditor() instanceof final ExtendedValueBoxEditor<?> valueBoxEditor
        && valueBoxEditor.getDecorator() != null) {
      final IsDecorator<?> decorator = valueBoxEditor.getDecorator();
      decorator.setFocusOnError(checkTime == CheckTimeEnum.ON_SUBMIT);
      editor = decorator;
    }
    initializeEditors(editor);

    return result;
  }

  /**
   * initialize one editor.
   *
   * @param editor editor to initialize
   */
  @SuppressWarnings({"unchecked"})
  public void initializeEditors(final Object editor) {
    if (editor instanceof final HasValueChangeHandlers hasValueChangeHandlers
        && valueChangeHandler != null) {
      hasValueChangeHandlers.addValueChangeHandler(valueChangeHandler);
      // if widget has a value change handler, validate on change
      if (validateOnVueChangeHandler != null) {
        hasValueChangeHandlers.addValueChangeHandler(validateOnVueChangeHandler);
      }
    }
    // if widget has a key up handler, validate on key up
    if (editor instanceof final HasKeyUpHandlers hasKeyUpHandlers
        && validateOnKeyUpHandler != null) {
      hasKeyUpHandlers.addKeyUpHandler(validateOnKeyUpHandler);
    }
    // try to submit form on return
    if (editor instanceof final HasKeyPressHandlers hasKeyPressHandlers
        && commitOnReturnHandler != null) {
      hasKeyPressHandlers.addKeyPressHandler(commitOnReturnHandler);
    }
  }
}
