package de.kah2.zodiac.libZodiac.planetary;

import de.kah2.zodiac.nova4jmt.JulianDay;
import de.kah2.zodiac.nova4jmt.api.LnDate;
import de.kah2.zodiac.nova4jmt.api.LnRstTime;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;

/**
 * This class contains rise- and set-times including time zone transformations
 * when we have UTC values generated by libNova.
 *
 * @author kahles
 */
public class ZonedRiseSet {

	private final LocalDateTime rise, set;
	private ZoneId localTimeZone;

	/**
	 * This constructor is used when rise and set times are calculated by
	 * libNova, which returns a {@link LnRstTime} object containing times in
	 * UTC.
	 */
	ZonedRiseSet(final LnRstTime lnRstTime, final ZoneId localTimeZone) {
		this.localTimeZone = localTimeZone;

		this.rise = this.getLocalDateTime(lnRstTime.rise);
		this.set = this.getLocalDateTime(lnRstTime.set);
	}

	/**
	 * This constructor instantiates this class based on {@link LocalDateTime}.
	 * No time zone transformations will be done.
	 * Only needed for testing purposes.
	 * @param rise an UTC rise time
	 * @param set an UTC set time
	 */
	public ZonedRiseSet(final LocalDateTime rise, final LocalDateTime set) {
		this.rise = rise;
		this.set = set;
	}

	public LocalDateTime getRise() {
		return this.rise;
	}

	public LocalDateTime getSet() {
		return this.set;
	}

	private LocalDateTime getLocalDateTime(final double julianDay) {

		final LnDate date = new LnDate();
		JulianDay.ln_get_date(julianDay, date);

		final ZonedDateTime utcZonedTime = LocalDateTime
				.of(date.years, date.months, date.days, date.hours, date.minutes)
				.atZone(PlanetaryDayData.TIME_ZONE_ID_LIBNOVA);
		return utcZonedTime.withZoneSameInstant(this.localTimeZone).toLocalDateTime();
	}

	@Override
	public String toString() {
		return "^" + this.rise + " v" + this.set;
	}
}
