package de.juplo.reactorm;

import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Implementation of the <code>Subscription</code> contract.
 * <em>Intended for internal usage only.</em>
 * @see Subscription
 * @author Kai Moritz
 */
public class PaginatedSourceSubscription implements Subscription
{
  public static final Logger LOG =
      LoggerFactory.getLogger(PaginatedSourceSubscription.class);


  private final PaginatedSourcePublisher publisher;
  private final Subscriber subscriber;


  PaginatedSourceSubscription(
      PaginatedSourcePublisher publisher,
      Subscriber subscriber
      )
  {
    this.publisher = publisher;
    this.subscriber = subscriber;
  }


  @Override
  public void request(long request)
  {
    if (LOG.isDebugEnabled())
      LOG.debug(
          "{}: request for {} items from subscriber {}",
          publisher,
          request,
          Integer.toHexString(subscriber.hashCode())
          );

    if (!publisher.isSubscribed(subscriber))
      // See: https://github.com/reactive-streams/reactive-streams-jvm/blob/v1.0.1/README.md#3.6
      return;

    if (publisher.isCompleted() || publisher.hasError())
    {
      publisher.publish();
      return;
    }

    if (request <= 0)
    {
      // See: https://github.com/reactive-streams/reactive-streams-jvm/blob/v1.0.1/README.md#3.9
      publisher.signalError(new IllegalArgumentException("non-positive request signals are illegal"));
      return;
    }

    publisher.addRequest(subscriber, request);
    publisher.publish();
  }

  @Override
  public void cancel()
  {
    if (LOG.isDebugEnabled())
      LOG.debug(
          "{}: subscriber {} canceled its sbscription!",
          publisher,
          Integer.toHexString(subscriber.hashCode())
          );
    publisher.unsubscribe(subscriber);
  }
}
