package de.juplo.reactorm;


import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Queue;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.function.BiFunction;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



/**
 * Implementation of the <code>Pulblisher</code> contract.
 * <p>
 * This implementation works against any <code>BiFunction</code> that provides
 * a <strong>pagination</strong> of a result by mapping two <code>Long</code>
 * values that represent the <strong>start</strong> index in the result list
 * and the <strong>limit</strong> for the current request to a list of results,
 * that represents the according part of the complete result.
 * </p>
 * <p>
 * Hence, stricly speeking it does not only allow to build a reactive stream
 * of an <code>ORM</code>-source, but of any source that can provide paginated
 * results.
 * </p>
 * <p>
 * The idea of this implementation is to request a page of results and buffer
 * it until it is fully consumed by the requests of the subscribers.
 * The parameters <code>max</code>, <code>min</code> and <code>fill</code> can
 * be used to tune the size of the buffered page that is requested from the
 * underlying source:
 * </p>
 * <ol>
 *   <li>
 *   <code>max</code> specifies the maximal size for a requested page
 *   (default is 100).
 *   </li>
 *   <li>
 *   <code>min</code> specifies the minimal size for a page that is requested
 *   from the underlying source. If less than <code>min</code> items are
 *   requested by the subscribers, at least <code>min</code> items will be
 *   requested (default is 100).
 *   </li>
 *   <li>
 *   <code>fill</code> specifies the number of items that should ideally be
 *   buffered at least, to minimize delays that may otherwise be caused by the
 *   underlying source. If less than <code>fill</code> items are available,
 *   the next page will be requested even if these remaining items are enough
 *   to satisfy all currently known requests (default is 20).
 *   </li>>
 * </ol>
 * <p>
 * This implementation serves all subscribers from one result.
 * No guarantees are given for the order in which the subscribers are served.
 * Hence, <em>if more than one subscriber is subscribed to this publisher, an
 * ordering of the result list will most probably get lost</em>.
 * </p>
 */
public class PaginatedSourcePublisher<T> implements Publisher<T>
{
  public static final Logger LOG =
      LoggerFactory.getLogger(PaginatedSourcePublisher.class);

  /** Default value for <code>max</code>. */
  public static final int DEFAULT_MAX = 100;
  /** Default value for <code>min</code>. */
  public static final int DEFAULT_MIN = 100;
  /** Default value for <code>fill</code>. */
  public static final int DEFAULT_FILL = 20;



  private final Executor executor;
  private final BiFunction<Long, Long, List<T>> function;
  private final int min;
  private final int max;
  private final int fill;

  private volatile boolean completed = false;
  private volatile Throwable error = null;

  private long current = 0l;
  private long start = 0l;
  private boolean publishing;

  private final Map<Subscriber<? super T>, Request> requests = new HashMap<>();
  private final Queue<T> unpublished = new LinkedBlockingQueue<>();


  /**
   * Convenient constructor that uses the defaults and uses an <code>Executor</code>
   * that schedules all incomming requests in the current thread.
   * <p>
   * This convenient constructor only requires the function that queries the
   * underlying source.
   * But be aware that
   * <strong>the resulting publisher will not comply with the reactive
   * streams specification in a strictly sense</strong>, because it does not
   * enforces the rule 3.3. Therefore, a stack overflow may be caused by a
   * publisher that was constructed through this convenient constructor in
   * some circumstances.
   * </p>
   * <p>
   * If a stack overflow is caused, depends on the usage circumstances. In a
   * log of simple scenarios it is totaly safe to use a publisher, that was
   * constructed through this constructor.
   * </p>
   * @see https://github.com/reactive-streams/reactive-streams-jvm/blob/v1.0.1/README.md#3.3
   * @param function the function used to access the underlying source
   */
  public PaginatedSourcePublisher(final BiFunction<Long ,Long, List<T>> function)
  {
    this(function, DEFAULT_MIN, DEFAULT_MAX, DEFAULT_FILL);
  }

  /**
   * Convenient constructor that only requires the function to acces the
   * underlying source and an <code>Executor</code> used for scheduling
   * incomming requests.
   * <p>
   * A publisher, that was constructed through this constructor is guaranteed
   * to fully comply with the reactive streams specification, if the supplied
   * <code>Executor</code> uses a different thread then the current one to
   * schedule its tasks.
   * </p>
   * @see https://github.com/reactive-streams/reactive-streams-jvm/blob/v1.0.1/README.md
   * @param executor the executor used to schedule incomming requests
   * @param function the function used to access the underlying source
   */
  public PaginatedSourcePublisher(
      final Executor executor,
      final BiFunction<Long ,Long, List<T>> function
      )
  {
    this(executor, function, DEFAULT_MIN, DEFAULT_MAX, DEFAULT_FILL);
  }

  /**
   * A configurable convenient constructor that uses an <code>Executor</code>
   * that schedules all incomming requests in the current thread.
   * <p>
   * Be aware that
   * <strong>the resulting publisher will not comply with the reactive
   * streams specification in a strictly sense</strong>, because it does not
   * enforces the rule 3.3. Therefore, a stack overflow may be caused by a
   * publisher that was constructed through this convenient constructor in
   * some circumstances.
   * </p>
   * <p>
   * If a stack overflow is caused, depends on the usage circumstances. In a
   * log of simple scenarios it is totaly safe to use a publisher, that was
   * constructed through this constructor.
   * </p>
   * @see https://github.com/reactive-streams/reactive-streams-jvm/blob/v1.0.1/README.md#3.3
   * @param function the function used to access the underlying source
   * @param min the minimum page-size to request
   * @param max the maximum page-size to request
   * @param fill the number of available items, that will trigger a new request
   */
  public PaginatedSourcePublisher(
      final BiFunction<Long ,Long, List<T>> function,
      final int min,
      final int max,
      final int fill
      )
  {
    this(Executors.newSingleThreadExecutor(), function, min, max, fill);
  }

  /**
   * Fully configurable constructor.
   * <p>
   * A publisher, that was constructed through this constructor is guaranteed
   * to fully comply with the reactive streams specification, if the supplied
   * <code>Executor</code> uses a different thread then the current one to
   * schedule its tasks.
   * </p>
   * @see https://github.com/reactive-streams/reactive-streams-jvm/blob/v1.0.1/README.md
   * @param executor the executor used to schedule incomming requests
   * @param function the function used to access the underlying source
   * @param min the minimum page-size to request
   * @param max the maximum page-size to request
   * @param fill the number of available items, that will trigger a new request
   */
  public PaginatedSourcePublisher(
      final Executor executor,
      final BiFunction<Long ,Long, List<T>> function,
      final int min,
      final int max,
      final int fill
      )
  {
    this.executor = new SerialExecutor(executor);
    this.function = function;
    if (min < 1)
      throw new IllegalArgumentException("Minimum chunk size must be at least one!");
    this.min = min;
    if (max < 1)
      throw new IllegalArgumentException("Maximum chunk size must be at least one!");
    this.max = max;
    if (fill < 0)
      throw new IllegalArgumentException("Minimum queue length must be at least zero!");
    if (fill > max/2)
      throw new IllegalArgumentException("Minimum queue length can be at most the half of the maximum chunk size!");
    this.fill = fill;
  }


  @Override
  public void subscribe(Subscriber<? super T> subscriber)
  {
    if (LOG.isDebugEnabled())
      LOG.debug(
          "{}: subscription from {}",
          this,
          Integer.toHexString(subscriber.hashCode())
          );

    Subscription subscription = new PaginatedSourceSubscription(this, subscriber);

    try
    {
      synchronized (requests) { requests.put(subscriber, new Request()); }
      subscriber.onSubscribe(subscription);
      // Be sure that an onError() is emitted imediately, if the publisher
      // already is in an error-state! See:
      // https://github.com/reactive-streams/reactive-streams-jvm/blob/v1.0.1/README.md#1.4
      publish();
    }
    catch (NullPointerException e)
    {
      throw e;
    }
    catch (Exception e)
    {
      LOG.debug("{}: unexpected error {}", this, e);
      error = e;
      publish();
    }

    if (LOG.isDebugEnabled())
      LOG.debug(
          "{}: subscription from {} done!",
          this,
          Integer.toHexString(subscriber.hashCode())
          );
  }


  /**
   * Unsubscribe the provided subscriber.
   * @param subscriber the subscriber to unsubscribe.
   */
  protected void unsubscribe(Subscriber subscriber)
  {
    synchronized (requests) { requests.remove(subscriber); }
  }

  /**
   * Reports, if the provided subscriber is subscribed to this publisher.
   * @param subscriber the subscriber to check.
   * @return <code>true</code> if the subscriber is subscribed to this publisher.
   */
  public boolean isSubscribed(Subscriber subscriber)
  {
    synchronized (requests) { return requests.containsKey(subscriber); }
  }

  /**
   * Adds an additional request to the subscriber.
   * The underlying helper class ensures, that the subscirber is marked to
   * request an invinite amount of items, if the sum exceeds
   * <code>Long.MAX_VALUE</code>.
   * @param subscriber the subscriber to add the reqeust to.
   * @param request the number of additionally requested items.
   */
  protected void addRequest(Subscriber subscriber, long request)
  {
    synchronized (requests) { requests.get(subscriber).add(request); }
  }

  /**
   * Reports, if this publisher is completed.
   * @return <code>true</code> if this publisher is completed.
   */
  public boolean isCompleted()
  {
    return completed && unpublished.isEmpty();
  }

  /**
   * Reports, if this publisher has encountered an error.
   * @return <code>true</code>, if this publisher has encounted an error.
   */
  public boolean hasError()
  {
    return error != null;
  }

  /**
   * Mark this publisher as errounous.
   * @param error the throwable, that represents the encountered error.
   */
  protected void signalError(Throwable error)
  {
    this.error = error;
    publish();
  }

  /**
   * Publish currently available items to the subscribers, that signaled need
   * and schedule requests to the underlying source for more items, if
   * necessary.
   */
  protected synchronized void publish()
  {
    if (error != null)
    {
      synchronized (requests)
      {
        requests.keySet().forEach(subscriber -> subscriber.onError(error));
        requests.clear();
      }
      return;
    }

    if (publishing)
      return;

    publishing = true;
    int published = Integer.MAX_VALUE;
    while (published > 0 && !unpublished.isEmpty())
    {
      published = 0;
      synchronized (requests)
      {
        for (Entry<Subscriber<? super T>, Request> request : requests.entrySet())
        {
          if (unpublished.isEmpty())
          {
            LOG.debug("{}: no more itemes available for publication", this);
            break;
          }
          if (!request.getValue().consume())
          {
            if (LOG.isDebugEnabled())
              LOG.debug(
                  "{}: request from subscriber {} is satisfied!",
                  this,
                  Integer.toHexString(request.getKey().hashCode())
                  );
            continue;
          }
          T item = unpublished.remove();
          if (LOG.isDebugEnabled())
            LOG.debug(
                "{}: calling {}.onNext({}), remaining requested items for this subscriber: {}, available: {}",
                this,
                Integer.toHexString(request.getKey().hashCode()),
                item,
                request.getValue(),
                unpublished.size()
                );
          current++;
          published++;
          try
          {
            request.getKey().onNext(item);
          }
          catch (Throwable t)
          {
            LOG.error(
                "unallowed error while signalling onNext({}) to subscriber {}",
                item,
                Integer.toHexString(request.getKey().hashCode())
                );
            signalError(t);
          }
        }
      }
    }
    publishing = false;

    if (!unpublished.isEmpty())
      return;

    if (completed)
    {
      synchronized (requests)
      {
        requests.keySet().forEach(subscriber -> subscriber.onComplete());
        requests.clear();
      }
      return;
    }

    long requested;
    synchronized (requests)
    {
      requested =
          requests
              .values()
              .stream()
              .map(request -> request.get())
              .reduce(0l, (acc, val) ->
              {
                long sum = acc + val;
                return sum > max || (sum <= 0 && val > 0) ? max : sum;
              });
    }

    if (requested == 0)
      return;

    long first = start;
    long limit = requested < min ? min : requested;
    if (current + requested < start - fill)
    {
      LOG.debug(
          "{}: no need to request more items (published={}, requested={}, fetched={}, minfill={})",
          this,
          current,
          requested,
          first,
          fill
          );
      return;
    }
    LOG.debug(
        "{}: scheduling new fetch {} -> {} (published={}, requested={}, fetched={}, minfill={})",
        this,
        first + 1,
        first + limit,
        current,
        requested,
        first,
        fill
        );
    start += limit;
    executor.execute(new Runnable()
    {
      @Override
      public void run()
      {
        try
        {
          LOG.debug(
              "{}: fetching {} -> {}",
              PaginatedSourcePublisher.this,
              first +1,
              first + limit
              );
          List<T> found = function.apply(first, limit);
          LOG.debug(
              "{}: fetched {}",
              PaginatedSourcePublisher.this,
              found.size()
              );
          if (found.size() < limit)
          {
            LOG.debug(
                "{}: DONE (no more events available)!",
                PaginatedSourcePublisher.this
                );
            completed = true;
          }
          unpublished.addAll(found);
        }
        catch (Exception e)
        {
          error = e;
        }
        finally
        {
          publish();
        }
      }
    });
  }

  @Override
  public String toString()
  {
    return Integer.toHexString(hashCode());
  }
}
