package de.julielab.concepts.db.creators;

import com.google.gson.Gson;
import de.julielab.bioportal.ontologies.data.OntologyClass;
import de.julielab.bioportal.util.BioPortalToolUtils;
import de.julielab.concepts.db.core.DefaultFacetCreator;
import de.julielab.concepts.db.core.spi.ConceptCreator;
import de.julielab.concepts.util.ConceptCreationException;
import de.julielab.concepts.util.ConceptDBManagerRuntimeException;
import de.julielab.java.utilities.FileUtilities;
import de.julielab.neo4j.plugins.datarepresentation.*;
import de.julielab.neo4j.plugins.datarepresentation.constants.FacetConstants;
import org.apache.commons.configuration2.HierarchicalConfiguration;
import org.apache.commons.configuration2.ex.ConfigurationException;
import org.apache.commons.configuration2.tree.ImmutableNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static de.julielab.concepts.db.core.ConfigurationConstants.*;
import static de.julielab.java.utilities.ConfigurationUtilities.*;

/**
 * <p>
 * Creates {@link ImportConcept} instances from the ontology class output
 * created by the julielab-bioportal-ontology-tools.
 * </p>
 * <p>
 * This class reads a specific JSON format as it is generated by the
 * julielab-bioportal-ontology-tools. The JSON format is centered around the
 * human-readable names of ontology classes and their taxonomical structure. For
 * more information, refer to the link given below.
 * </p>
 *
 * @author faessler
 * @see <url>https://github.com/JULIELab/julielab-bioportal-ontology-tools</url>
 */
public class JulielabBioPortalToolsConceptCreator implements ConceptCreator {


    private static final Logger log = LoggerFactory.getLogger(JulielabBioPortalToolsConceptCreator.class);

    @Override
    public Stream<ImportConcepts> createConcepts(HierarchicalConfiguration<ImmutableNode> config)
            throws ConceptCreationException {
        String facetGroupNameKey = slash(FACET, CREATOR, CONFIGURATION, FACET_GROUP, NAME);
        String ontologiesPathKey = slash(CONCEPTS, CREATOR, CONFIGURATION, PATH);
        try {
            checkParameters(config, facetGroupNameKey,
                    ontologiesPathKey);
            checkFilesExist(config, ontologiesPathKey);
        } catch (ConfigurationException e) {
            throw new ConceptCreationException(e);
        }
        // First, read the configuration.
        String facetGroupName = config.getString(facetGroupNameKey);
        File ontologyNamesPath = new File(config.getString(ontologiesPathKey));

        Gson gson = new Gson();
        // First, get all files with class names.
        File[] ontologyNameFiles;
        if (ontologyNamesPath.isDirectory())
            ontologyNameFiles = ontologyNamesPath
                    .listFiles((f, n) -> n.endsWith(".jsonlst") || n.endsWith(".jsonlst.gz"));
        else
            ontologyNameFiles = new File[]{ontologyNamesPath};

        log.info("Reading {} ontology class files for concept creation.", ontologyNameFiles.length);
        // The following stream reads the class name files one after the other and
        // creates a facet for each file and the respective concepts. Thus, the
        // assumption is that each file corresponds to one ontology.
        // Also, the file name must be a unique name for the facet.
        return Stream.of(ontologyNameFiles).map(f -> {
            try {
                String acronym = BioPortalToolUtils.getAcronymFromFileName(f);

                // The format of the name files is one class per line as
                // a JSON object on its own. We will now build a JSON
                // array out of all the classes
                // of the file
                BufferedReader br = FileUtilities.getReaderFromFile(f);
                // Convert the JSON lines to OntologyClass objects. However, the OntologyClass
                // class is the representation for ontology classes of the BioPortal tools
                // project. We need to translate this to ImportConcept instances.
                Stream<OntologyClass> classStream = br.lines().map(l -> gson.fromJson(l, OntologyClass.class));
                // Convert the OntologyClass objects to ImportConcepts
                Stream<ImportConcept> conceptStream = classStream.map(c -> {
                    // Collect the parents of the current class, if there are any.
                    List<ConceptCoordinates> parentCoordinates = Collections.emptyList();
                    if (c.parents != null && c.parents.parents != null)
                        parentCoordinates = c.parents.parents.stream()
                                .map(p -> new ConceptCoordinates(p, acronym, true)).collect(Collectors.toList());
                    // Now create the actual concepts that can be imported into the database.
                    return new ImportConcept(c.prefLabel, c.synonym.synonyms, c.definition,
                            new ConceptCoordinates(c.id, acronym, true), parentCoordinates);
                });
                // Note: Facet groups are unique by name in the database (the
                // ConceptManager that does the concept insertion makes sure of it).
                ImportFacetGroup fg = new ImportFacetGroup(facetGroupName);
                // We use the acronym as name, short name and custom ID. The last shouldn't be a
                // problem since ontology acronyms are unique within BioPortal. There could be
                // an issue with ontologies from outside, but let's just hope for now that there
                // won't.
                ImportFacet facet = new ImportFacet(fg, acronym, acronym, acronym,
                        FacetConstants.SRC_TYPE_HIERARCHICAL);
                boolean noFacet = config.getBoolean(slash(FACET, CREATOR, REQUEST, DefaultFacetCreator.NO_FACET), false);
                facet.setNoFacet(noFacet);
                String[] labels = config.getStringArray(slash(FACET, CREATOR, REQUEST, DefaultFacetCreator.LABELS));
                if (labels != null && labels.length > 0)
                    facet.setLabels(Arrays.asList(labels));
                return new ImportConcepts(conceptStream, facet);
            } catch (IOException e) {
                throw new ConceptDBManagerRuntimeException(new ConceptCreationException(e));
            }
        });
    }


    @Override
    public void exposeParameters(String basePath, HierarchicalConfiguration<ImmutableNode> template) {
        template.addProperty(slash(basePath, CONCEPTS, CREATOR, NAME), getName());
        template.addProperty(slash(basePath, CONCEPTS, CREATOR, REQUEST, PATH), "");
        template.setProperty(slash(basePath, FACET, CREATOR, REQUEST, FACET_GROUP, NAME), "Ontologies");
        template.setProperty(slash(basePath, FACET, CREATOR, REQUEST, DefaultFacetCreator.LABELS), "");
    }

    @Override
    public String getName() {
        return "JulielabBioPortalToolsConceptCreator";
    }
}
