/*
 * Decompiled with CFR 0.152.
 */
package de.julielab.bioportal.ontologies;

import com.google.gson.Gson;
import de.julielab.bioportal.ontologies.AnnotationPropertySet;
import de.julielab.bioportal.ontologies.OntologyLoader;
import de.julielab.bioportal.ontologies.data.OntologyClass;
import de.julielab.bioportal.ontologies.data.OntologyClassParents;
import de.julielab.bioportal.ontologies.data.OntologyClassSynonyms;
import de.julielab.bioportal.util.BioPortalToolUtils;
import de.julielab.java.utilities.FileUtilities;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.semanticweb.HermiT.ReasonerFactory;
import org.semanticweb.owlapi.io.UnparsableOntologyException;
import org.semanticweb.owlapi.model.OWLAnnotation;
import org.semanticweb.owlapi.model.OWLAnnotationProperty;
import org.semanticweb.owlapi.model.OWLAnnotationValue;
import org.semanticweb.owlapi.model.OWLClass;
import org.semanticweb.owlapi.model.OWLClassExpression;
import org.semanticweb.owlapi.model.OWLEntity;
import org.semanticweb.owlapi.model.OWLLiteral;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.OWLOntologyCreationException;
import org.semanticweb.owlapi.model.parameters.Imports;
import org.semanticweb.owlapi.reasoner.OWLReasoner;
import org.semanticweb.owlapi.reasoner.OWLReasonerFactory;
import org.semanticweb.owlapi.search.EntitySearcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OntologyClassNameExtractor {
    private static final Logger log = LoggerFactory.getLogger(OntologyClassNameExtractor.class);
    private static final Logger logUnparsableOntologies = LoggerFactory.getLogger((String)(OntologyClassNameExtractor.class.getCanonicalName() + ".unparsable"));
    private Gson gson = BioPortalToolUtils.getGson();
    private ExecutorService executor;
    private OWLReasonerFactory reasonerFactory;
    private boolean filterDeprecated;

    public OntologyClassNameExtractor() {
        this(Executors.newFixedThreadPool(4), false, false);
    }

    public OntologyClassNameExtractor(ExecutorService executor, boolean applyReasoning) {
        this(executor, applyReasoning, false);
    }

    public OntologyClassNameExtractor(ExecutorService executor, boolean applyReasoning, boolean filterDeprecated) {
        this.executor = executor;
        if (applyReasoning) {
            this.reasonerFactory = new ReasonerFactory();
        }
        this.filterDeprecated = filterDeprecated;
    }

    public int run(File input, File submissionsDirectory, File output) throws InterruptedException, ExecutionException {
        return this.run(input, submissionsDirectory, output, null);
    }

    public int run(File input, File submissionsDirectory, File outputDir, Set<String> ontologiesToExtract) throws InterruptedException, ExecutionException {
        if (!outputDir.exists()) {
            outputDir.mkdirs();
        }
        int numOntologies = 0;
        if (!input.isDirectory()) {
            log.error("{} is not a directory. Please specify the directory containing the ontology files.", (Object)input);
            return 0;
        }
        if (outputDir.isFile()) {
            throw new IllegalArgumentException("The output path \"" + outputDir.getAbsolutePath() + "\" is a file but should be a directory.");
        }
        if (!outputDir.exists()) {
            log.debug("Creating output directory \"{}\"", (Object)outputDir);
            outputDir.mkdirs();
        }
        if (ontologiesToExtract != null && !ontologiesToExtract.isEmpty()) {
            log.info("Extracting class names for ontologies with acronyms {} in directory {}", ontologiesToExtract, (Object)input);
        } else {
            log.info("Extracting class names for all ontologies in {}", (Object)input);
        }
        File[] files = input.listFiles();
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>(files.length);
        for (int i = 0; i < files.length; ++i) {
            File file = files[i];
            if (ontologiesToExtract != null && !ontologiesToExtract.isEmpty() && !ontologiesToExtract.contains(BioPortalToolUtils.getAcronymFromFileName(file))) continue;
            Future<Void> future = this.executor.submit(new NameExtractorWorker(file, submissionsDirectory, input, outputDir));
            futures.add(future);
            ++numOntologies;
        }
        for (Future future : futures) {
            future.get();
        }
        log.info("Shutting down executor service.");
        this.executor.shutdown();
        return numOntologies;
    }

    private void extractNamesForOntology(File ontologyFileOrDirectory, File submissionsDirectory, File outputDir, OntologyLoader ontologyLoader) throws IOException, OWLOntologyCreationException {
        OWLOntology o;
        log.info("Processing file or directory \"{}\"", (Object)ontologyFileOrDirectory);
        String acronym = BioPortalToolUtils.getAcronymFromFileName(ontologyFileOrDirectory);
        File submissionFile = new File(submissionsDirectory.getAbsolutePath() + File.separator + acronym + ".sub.json.gz");
        AnnotationPropertySet properties = new AnnotationPropertySet(ontologyLoader.getOntologyManager(), submissionFile);
        File classesFile = new File(outputDir.getAbsolutePath() + File.separator + acronym + ".cls.jsonlst.gz");
        if (classesFile.exists() && classesFile.length() > 0L) {
            log.info("Classes file {} already exists and is not empty. Not extracting class names again.", (Object)classesFile);
            return;
        }
        try {
            log.debug("Loading ontology from {} {}", (Object)(ontologyFileOrDirectory.isFile() ? "file" : "directory"), (Object)ontologyFileOrDirectory);
            o = ontologyLoader.loadOntology(ontologyFileOrDirectory);
            log.trace("Loading done for {}", (Object)ontologyFileOrDirectory);
        }
        catch (OWLOntologyCreationException e) {
            log.error("Error while loading ontology {}.", (Object)acronym);
            throw e;
        }
        OWLReasoner reasoner = this.reasonerFactory != null ? this.reasonerFactory.createReasoner(o) : null;
        log.debug("Writing extracted class names for ontology {} to {}", (Object)acronym, (Object)classesFile);
        this.writeNames(properties, classesFile, o, reasoner);
        ontologyLoader.clearLoadedOntologies();
    }

    private void writeNames(AnnotationPropertySet properties, File classesFile, OWLOntology o, OWLReasoner reasoner) throws IOException {
        try (BufferedOutputStream os = FileUtilities.getOutputStreamToFile((File)classesFile);){
            Stream classesInSignature = o.classesInSignature(Imports.INCLUDED);
            Iterator iterator = classesInSignature.iterator();
            while (iterator.hasNext()) {
                OWLClass c = (OWLClass)iterator.next();
                boolean obsolete = this.determineObsolete(o, c, properties);
                if (obsolete && this.filterDeprecated) {
                    log.trace("Excluding obsolete class {}", (Object)c.getIRI());
                    continue;
                }
                String preferredName = this.determinePreferredName(o, c, properties);
                OntologyClassSynonyms synonyms = this.determineSynonyms(o, c, properties);
                String definition = this.determineDefinition(o, c, properties);
                OntologyClassParents ontologyClassParents = this.determineClassParents(o, c, reasoner);
                OntologyClass ontologyClass = new OntologyClass();
                ontologyClass.id = c.getIRI().toString();
                ontologyClass.prefLabel = preferredName;
                if (obsolete) {
                    ontologyClass.obsolete = true;
                }
                if (synonyms.synonyms != null && !synonyms.synonyms.isEmpty()) {
                    ontologyClass.synonym = synonyms;
                }
                if (!StringUtils.isBlank((CharSequence)definition)) {
                    ontologyClass.definition = Arrays.asList(definition);
                }
                if (ontologyClassParents.parents != null && !ontologyClassParents.parents.isEmpty()) {
                    ontologyClass.parents = ontologyClassParents;
                }
                IOUtils.write((String)(this.gson.toJson((Object)ontologyClass) + "\n"), (OutputStream)os, (String)"UTF-8");
            }
        }
    }

    private OntologyClassParents determineClassParents(OWLOntology o, OWLClass c, OWLReasoner reasoner) {
        Stream<OWLClassExpression> superClasses = reasoner != null ? reasoner.getSuperClasses((OWLClassExpression)c, true).entities().map(OWLClassExpression.class::cast) : EntitySearcher.getSuperClasses((OWLClass)c, (OWLOntology)o);
        OntologyClassParents classParents = new OntologyClassParents();
        Iterator iterator = superClasses.iterator();
        while (iterator.hasNext()) {
            OWLClassExpression classExpr = (OWLClassExpression)iterator.next();
            if (classExpr.isAnonymous()) continue;
            OWLClass owlClass = classExpr.asOWLClass();
            classParents.addParent(owlClass.getIRI().toString());
        }
        return classParents;
    }

    private String determineDefinition(OWLOntology o, OWLClass c, AnnotationPropertySet properties) {
        for (OWLAnnotationProperty definitionProp : properties.getDefinitionProps()) {
            Stream definitionAnnotations = EntitySearcher.getAnnotations((OWLEntity)c, (OWLOntology)o, (OWLAnnotationProperty)definitionProp);
            Iterator iterator = definitionAnnotations.iterator();
            while (iterator.hasNext()) {
                OWLAnnotation owlAnnotation = (OWLAnnotation)iterator.next();
                OWLAnnotationValue value = owlAnnotation.getValue();
                String definition = value instanceof OWLLiteral ? ((OWLLiteral)value).getLiteral() : value.toString();
                if (StringUtils.isBlank((CharSequence)definition)) continue;
                return definition;
            }
        }
        return null;
    }

    private OntologyClassSynonyms determineSynonyms(OWLOntology o, OWLClass c, AnnotationPropertySet properties) {
        ArrayList<String> synonyms = new ArrayList<String>();
        for (OWLAnnotationProperty synonymProp : properties.getSynonymProps()) {
            Stream synonymAnnotations = EntitySearcher.getAnnotations((OWLEntity)c, (OWLOntology)o, (OWLAnnotationProperty)synonymProp);
            Iterator iterator = synonymAnnotations.iterator();
            while (iterator.hasNext()) {
                OWLAnnotation owlAnnotation = (OWLAnnotation)iterator.next();
                OWLAnnotationValue value = owlAnnotation.getValue();
                String synonym = value instanceof OWLLiteral ? ((OWLLiteral)value).getLiteral() : value.toString();
                if (StringUtils.isBlank((CharSequence)synonym)) continue;
                synonyms.add(synonym);
            }
        }
        OntologyClassSynonyms classSynonyms = new OntologyClassSynonyms();
        classSynonyms.synonyms = synonyms;
        return classSynonyms;
    }

    private String determinePreferredName(OWLOntology o, OWLClass c, AnnotationPropertySet properties) {
        for (OWLAnnotationProperty prefNameProp : properties.getPrefNameProps()) {
            Stream prefNameAnnotations = EntitySearcher.getAnnotations((OWLEntity)c, (OWLOntology)o, (OWLAnnotationProperty)prefNameProp);
            Iterator iterator = prefNameAnnotations.iterator();
            while (iterator.hasNext()) {
                String preferredName;
                OWLAnnotation owlAnnotation = (OWLAnnotation)iterator.next();
                OWLAnnotationValue value = owlAnnotation.getValue();
                if (value instanceof OWLLiteral) {
                    preferredName = ((OWLLiteral)value).getLiteral();
                    if (preferredName.isEmpty()) {
                        continue;
                    }
                } else {
                    preferredName = value.toString();
                }
                return preferredName;
            }
        }
        return c.getIRI().getRemainder().orElse(c.getIRI().getIRIString());
    }

    private boolean determineObsolete(OWLOntology o, OWLClass c, AnnotationPropertySet properties) {
        boolean isObsolete = false;
        for (OWLAnnotationProperty obsoleteProp : properties.getObsoleteProps()) {
            Stream obsoleteAnnotations = EntitySearcher.getAnnotations((OWLEntity)c, (OWLOntology)o, (OWLAnnotationProperty)obsoleteProp);
            Iterator iterator = obsoleteAnnotations.iterator();
            while (iterator.hasNext()) {
                OWLAnnotation owlAnnotation = (OWLAnnotation)iterator.next();
                Optional optLiteral = owlAnnotation.getValue().asLiteral();
                if (!optLiteral.isPresent()) continue;
                String literal = ((OWLLiteral)optLiteral.get()).getLiteral().toLowerCase();
                if (!literal.equals("true") && !literal.equals("false")) {
                    log.warn("The obsolete property value of class {} of ontology {} is neither true nor false", (Object)c.getIRI(), (Object)o.getOntologyID());
                }
                isObsolete |= Boolean.parseBoolean(literal);
            }
        }
        return isObsolete;
    }

    public void shutDown() {
        this.executor.shutdown();
    }

    private class NameExtractorWorker
    implements Callable<Void> {
        private File file;
        private File submissionsDirectory;
        private File outputDir;
        private OntologyLoader ontologyLoader;
        private File ontosDir;

        public NameExtractorWorker(File file, File submissionsDirectory, File ontosDir, File outputDir) {
            this.file = file;
            this.submissionsDirectory = submissionsDirectory;
            this.ontosDir = ontosDir;
            this.outputDir = outputDir;
            this.ontologyLoader = new OntologyLoader();
        }

        @Override
        public Void call() throws Exception {
            block10: {
                if (BioPortalToolUtils.isSupportedOntologyFile(this.file) || this.file.isDirectory() && this.file.listFiles((f, name) -> name.equals("downloadFileName.txt")).length == 1) {
                    try {
                        OntologyClassNameExtractor.this.extractNamesForOntology(this.file, this.submissionsDirectory, this.outputDir, this.ontologyLoader);
                    }
                    catch (UnparsableOntologyException e) {
                        log.error("Could not parse ontology file {}", (Object)this.file);
                        if (BioPortalToolUtils.isUMLSOntology(this.file)) {
                            log.warn("The unparsable ontology is in UMLS format. Those have sometimes issues by chemical names containing the character sequence ''' which is mistakenly interpreted as a string end quote for \"\"\" by the turtle parser in OWL API 5.x. It will be tried to remove such lines and then try parsing again.");
                            File backupDir = new File(this.ontosDir.getAbsoluteFile().getParentFile().getAbsolutePath() + File.separator + this.ontosDir.getName() + "-backup");
                            if (!backupDir.exists()) {
                                backupDir.mkdir();
                            }
                            File backupFile = new File(backupDir.getAbsolutePath() + File.separator + this.file.getName());
                            log.info("Creating backup of file {} to {}", (Object)this.file, (Object)backupFile);
                            if (!backupFile.exists()) {
                                Files.copy(this.file.toPath(), backupFile.toPath(), new CopyOption[0]);
                            } else {
                                log.info("File {} already exists, skipping.", (Object)backupFile);
                            }
                            log.warn("Replacing file {} with a copy where lines in question have been removed. Please note that the origin file is overwritten.", (Object)this.file);
                            Files.delete(this.file.toPath());
                            AtomicInteger removedLines = BioPortalToolUtils.fixUmlsFile(backupFile, this.file);
                            log.info("{} lines have been removed from {}", (Object)removedLines, (Object)backupFile);
                            try {
                                OntologyClassNameExtractor.this.extractNamesForOntology(this.file, this.submissionsDirectory, this.outputDir, this.ontologyLoader);
                            }
                            catch (UnparsableOntologyException e2) {
                                log.error("Fixed file also couldn't be parsed. Deleting fixed file and giving up. The backup file is left at {}", (Object)backupFile);
                                logUnparsableOntologies.error("File: {}", (Object)this.file, (Object)e);
                                Files.delete(this.file.toPath());
                            }
                            break block10;
                        }
                        logUnparsableOntologies.error("File: {}", (Object)this.file, (Object)e);
                    }
                } else {
                    log.debug("Ignoring file \"{}\" because it doesn't look like an ontology file", (Object)this.file);
                }
            }
            return null;
        }
    }
}

