/*
 * Decompiled with CFR 0.152.
 */
package de.julielab.bioportal.ontologies;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import de.julielab.bioportal.ontologies.DownloadStats;
import de.julielab.bioportal.ontologies.HttpHandler;
import de.julielab.bioportal.ontologies.OntologyListRetriver;
import de.julielab.bioportal.ontologies.data.OntologyMetaData;
import de.julielab.bioportal.ontologies.data.Submission;
import de.julielab.bioportal.util.BioPortalOntologyToolsException;
import de.julielab.bioportal.util.BioPortalToolUtils;
import de.julielab.bioportal.util.OntologyFileNotAvailableException;
import de.julielab.bioportal.util.ResourceAccessDeniedException;
import de.julielab.bioportal.util.ResourceDownloadException;
import de.julielab.bioportal.util.ResourceNotFoundException;
import de.julielab.java.utilities.FileUtilities;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.nio.charset.Charset;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.HttpEntity;
import org.apache.http.ParseException;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OntologyDownloader {
    private static final Logger log = LoggerFactory.getLogger(OntologyDownloader.class);
    public static final String ONTOLOGY_LIST = "ONTOLOGY_LIST.gz";
    private static final String submissionInclude = "submissionId,ontology,released,contact,status,description,creationDate,version,publication,hasOntologyLanguage,homepage,documentation,synonymProperty,definitionProperty,prefLabelProperty,obsoleteProperty";
    private static final String latestSubmissionEndpointFmtString = "http://data.bioontology.org/ontologies/%s/latest_submission?include=submissionId,ontology,released,contact,status,description,creationDate,version,publication,hasOntologyLanguage,homepage,documentation,synonymProperty,definitionProperty,prefLabelProperty,obsoleteProperty";
    private HttpHandler httpHandler;
    private Gson gson;
    private OntologyListRetriver ontologyListRetriver;
    private String apiKey;
    private ExecutorService executor;
    private Matcher filenameHeaderMatcher;

    public OntologyDownloader(String apiKey) {
        this.apiKey = apiKey;
        this.httpHandler = new HttpHandler(apiKey);
        this.ontologyListRetriver = new OntologyListRetriver(this.httpHandler);
        this.gson = BioPortalToolUtils.getGson();
        this.executor = Executors.newFixedThreadPool(20);
        this.filenameHeaderMatcher = Pattern.compile(".*filename=\"([^\"]+)\".*").matcher("");
    }

    public DownloadStats downloadOntologies(File ontologyDataDir, File ontologyInfoDir, Set<String> ontologiesToDownload) throws ParseException, IOException, BioPortalOntologyToolsException, InterruptedException, ExecutionException {
        DownloadStats downloadStats = new DownloadStats();
        if (!ontologyDataDir.exists()) {
            log.info("Ontology data directory {} does not exist and is created.", (Object)ontologyDataDir);
            ontologyDataDir.mkdirs();
        }
        if (!ontologyInfoDir.exists()) {
            log.info("Ontology information directory {} does not exist and is created.", (Object)ontologyInfoDir);
            ontologyInfoDir.mkdirs();
        }
        log.info("Downloading BioPortal ontologies to {}. {}.", (Object)ontologyDataDir, (Object)(ontologiesToDownload.size() == 0 ? "No restrictions on downloaded ontologies imposed" : "Ontology download is restricted to the ontologies with the following acronyms: " + StringUtils.join(ontologiesToDownload, ", ")));
        List<OntologyMetaData> ontologiesMetaData = this.ontologyListRetriver.getOntologiesMetaData(new File(ontologyInfoDir.getAbsolutePath() + File.separator + ONTOLOGY_LIST), ontologiesToDownload);
        ArrayList<ImmutablePair<Future<OntologyMetaData>, DownloadWorker>> ontologiesWithDownloadIssues = new ArrayList<ImmutablePair<Future<OntologyMetaData>, DownloadWorker>>();
        for (OntologyMetaData metaData : ontologiesMetaData) {
            if (metaData.summaryOnly) {
                log.debug("Skipping ontology {} because it is just a summary.", (Object)metaData.acronym);
                downloadStats.incNumSummaries();
                continue;
            }
            if (!metaData.type.equals("http://data.bioontology.org/metadata/Ontology")) {
                log.warn("Ontology {} has type {}", (Object)metaData.acronym, (Object)metaData.type);
            }
            DownloadWorker worker = new DownloadWorker(metaData, ontologyDataDir, ontologyInfoDir, downloadStats);
            try {
                worker.download();
            }
            catch (JsonSyntaxException | IOException e) {
                e.printStackTrace();
            }
            catch (ResourceDownloadException e) {
                log.info("Couldn't download all data for {} at first try. Pushing ontology to background and continuing with next ontology on the main thread.", (Object)metaData.acronym);
                Future<OntologyMetaData> future = this.executor.submit(worker);
                ontologiesWithDownloadIssues.add(new ImmutablePair<Future<OntologyMetaData>, DownloadWorker>(future, worker));
            }
        }
        log.info("Finished downloading ontologies in main thread. {} ontologies had issues during download. Waiting for them to finish.", (Object)ontologiesWithDownloadIssues.size());
        Iterator futureIt = ontologiesWithDownloadIssues.iterator();
        while (futureIt.hasNext()) {
            Pair futurePair = (Pair)futureIt.next();
            Future future = (Future)futurePair.getLeft();
            DownloadWorker worker = (DownloadWorker)futurePair.getRight();
            log.info("Waiting for ontology\u00a0{}", (Object)worker.getOntologyMetaData().acronym);
            OntologyMetaData metaData = (OntologyMetaData)future.get();
            if (metaData != null) {
                log.info("Ontology {} could be downloaded successfully. Removing from list of download issues.");
                downloadStats.removeOntologyWithDownloadError(metaData.acronym);
                futureIt.remove();
                continue;
            }
            log.info("Ontology {} failed to download.", (Object)worker.getOntologyMetaData().acronym);
            worker.removeOntologyFiles();
        }
        return downloadStats;
    }

    private String downloadInfoForOntology(String address, File destFile, OntologyMetaData metaData, String infoType) throws ResourceAccessDeniedException, ResourceNotFoundException, ResourceDownloadException, ParseException, IOException {
        if (destFile.exists() && destFile.length() > 0L) {
            log.info("The file {} exists and is not empty. It is kept, download of the file is skipped.", (Object)destFile);
            return IOUtils.toString((InputStream)FileUtilities.getInputStreamFromFile(destFile), Charset.forName("UTF-8"));
        }
        try {
            log.debug("Fetching {} from BioPortal for {}", (Object)infoType, (Object)metaData.acronym);
            HttpEntity propertiesResponse = this.httpHandler.sendGetRequest(address);
            String infoString = EntityUtils.toString(propertiesResponse);
            try (BufferedWriter w = FileUtilities.getWriterToFile(destFile);){
                w.write(infoString);
            }
            return infoString;
        }
        catch (ResourceDownloadException e) {
            log.error("Error occured when trying to retrieve latest " + infoType + " of ontology " + metaData.acronym + ":", e);
            throw e;
        }
    }

    private void downloadOntologyFile(File ontologyDataDir, OntologyMetaData ontoInf, Submission submission) throws IOException, OntologyFileNotAvailableException {
        int responseCode;
        String ontoLanguage = "unknown";
        if (submission.hasOntologyLanguage != null) {
            ontoLanguage = submission.hasOntologyLanguage.toLowerCase();
        }
        String fileName = ontoInf.acronym + "." + ontoLanguage + ".gz";
        File ontologyFile = new File(ontologyDataDir.getAbsolutePath() + File.separator + fileName);
        if (ontologyFile.exists() && ontologyFile.length() > 0L) {
            log.info("Ontology file {} exists and is not empty. File is kept and not downloaded again.", (Object)ontologyFile.getAbsolutePath());
            return;
        }
        File ontologyDir = new File(ontologyDataDir.getAbsolutePath() + File.separator + ontoInf.acronym);
        if (ontologyDir.exists()) {
            String[] list = ontologyDir.list((dir, name) -> !name.equals(".DS_Store"));
            if (null != list && list.length != 0) {
                log.info("Ontology directory {} exists and is not empty. The directory and its files are kept and not downloaded again.", (Object)ontologyDir);
            }
            return;
        }
        log.debug("Downloading ontology {} from {}.", (Object)ontoInf.acronym, (Object)ontoInf.links.download);
        HttpURLConnection conn = (HttpURLConnection)ontoInf.links.download.openConnection();
        conn.setRequestProperty("Authorization", "apikey token=" + this.apiKey);
        String mimeType = conn.getHeaderField("Content-Disposition");
        String downloadFileName = null;
        if (mimeType != null) {
            this.filenameHeaderMatcher.reset(mimeType);
            if (this.filenameHeaderMatcher.find()) {
                downloadFileName = this.filenameHeaderMatcher.group(1);
            }
        }
        if ((responseCode = conn.getResponseCode()) == 200) {
            try (InputStream is = conn.getInputStream();){
                if (null != downloadFileName && downloadFileName.toLowerCase().endsWith(".zip")) {
                    log.info("Download for ontology {} is a zip file. Storing the contents of the archive in directory {}.", (Object)ontoInf.acronym, (Object)ontologyDir);
                    new File(ontologyDataDir.getAbsolutePath() + File.separator + ontoInf.acronym).mkdir();
                    ZipInputStream zipStream = new ZipInputStream(is, Charset.forName("UTF-8"));
                    ZipEntry entry = zipStream.getNextEntry();
                    int numEntries = 0;
                    File outputFile = null;
                    while (entry != null) {
                        ++numEntries;
                        String filename = entry.getName();
                        outputFile = new File(ontologyDataDir.getAbsolutePath() + File.separator + ontoInf.acronym + File.separator + filename + ".gz");
                        if (!outputFile.getAbsoluteFile().getParentFile().exists()) {
                            outputFile.getAbsoluteFile().getParentFile().mkdirs();
                        }
                        this.writeStreamToFile(zipStream, outputFile);
                        entry = zipStream.getNextEntry();
                    }
                    if (numEntries == 1) {
                        log.info("Downloaded ZIP file {} for ontology {} only contained a single entry. Moving it to {}", downloadFileName, ontoInf.acronym, ontologyFile);
                        Files.move(outputFile.toPath(), ontologyFile.toPath(), new CopyOption[0]);
                        Files.delete(ontologyDir.toPath());
                    }
                    Files.write(Paths.get(ontologyDataDir.getAbsolutePath() + File.separator + ontoInf.acronym + File.separator + "downloadFileName.txt", new String[0]), downloadFileName.getBytes(), new OpenOption[0]);
                }
                this.writeStreamToFile(is, ontologyFile);
            }
        } else {
            throw new OntologyFileNotAvailableException("Ontology with acronym " + ontoInf.acronym + " does not yet have a file ready for download (error message: " + IOUtils.toString(conn.getErrorStream(), Charset.forName("UTF-8")) + ").");
        }
        conn.disconnect();
    }

    private void writeStreamToFile(InputStream is, File outputFile) throws FileNotFoundException, IOException {
        try (GZIPOutputStream os = new GZIPOutputStream(new FileOutputStream(outputFile));){
            int bytesRead = -1;
            byte[] buffer = new byte[4096];
            while ((bytesRead = is.read(buffer)) != -1) {
                ((OutputStream)os).write(buffer, 0, bytesRead);
            }
        }
    }

    private class DownloadWorker
    implements Callable<OntologyMetaData> {
        private File submissionFile;
        private File submissionsFile;
        private File projectsFile;
        private File analyticsFile;
        private OntologyMetaData metaData;
        private File ontologyDataDir;
        private DownloadStats downloadStats;
        private File metaDataFile;

        public DownloadWorker(OntologyMetaData metaData, File ontologyDataDir, File ontologyInfoDir, DownloadStats downloadStats) {
            this.metaData = metaData;
            this.ontologyDataDir = ontologyDataDir;
            this.downloadStats = downloadStats;
            this.metaDataFile = new File(ontologyInfoDir.getAbsolutePath() + File.separator + metaData.acronym + ".meta.json" + ".gz");
            this.submissionFile = new File(ontologyInfoDir.getAbsolutePath() + File.separator + metaData.acronym + ".sub.json" + ".gz");
            this.submissionsFile = new File(ontologyInfoDir.getAbsolutePath() + File.separator + metaData.acronym + ".subs.json" + ".gz");
            this.projectsFile = new File(ontologyInfoDir.getAbsolutePath() + File.separator + metaData.acronym + ".pro.json" + ".gz");
            this.analyticsFile = new File(ontologyInfoDir.getAbsolutePath() + File.separator + metaData.acronym + ".ana.json" + ".gz");
        }

        public OntologyMetaData getOntologyMetaData() {
            return this.metaData;
        }

        public void download() throws JsonSyntaxException, IOException, ResourceDownloadException, ParseException {
            try {
                if (!this.metaDataFile.exists()) {
                    try (BufferedWriter w = FileUtilities.getWriterToFile(this.metaDataFile);){
                        w.write(OntologyDownloader.this.gson.toJson(this.metaData));
                    }
                } else {
                    log.info("Meta data file {} already exist and is not overwritten", (Object)this.metaDataFile);
                }
                String submission = OntologyDownloader.this.downloadInfoForOntology(String.format(OntologyDownloader.latestSubmissionEndpointFmtString, this.metaData.acronym), this.submissionFile, this.metaData, "latest submission");
                OntologyDownloader.this.downloadInfoForOntology(this.metaData.links.submissions.toString(), this.submissionsFile, this.metaData, "submissions");
                OntologyDownloader.this.downloadInfoForOntology(this.metaData.links.projects.toString(), this.projectsFile, this.metaData, "projects");
                OntologyDownloader.this.downloadInfoForOntology(this.metaData.links.analytics.toString(), this.analyticsFile, this.metaData, "analytics");
                OntologyDownloader.this.downloadOntologyFile(this.ontologyDataDir, this.metaData, OntologyDownloader.this.gson.fromJson(submission, Submission.class));
                this.downloadStats.addDownloadedOntology(this.metaData.acronym);
            }
            catch (OntologyFileNotAvailableException e) {
                log.warn("Ontology {} could not be downloaded because no file is available for download. Deleting info files for this ontology.", (Object)this.metaData.acronym);
                this.downloadStats.addOntologyWithoutFile(this.metaData.acronym);
                this.removeOntologyFiles();
            }
            catch (ResourceAccessDeniedException e) {
                log.warn("Ontology {} could not be downloaded because the server rejected access: {}", (Object)this.metaData.acronym, (Object)e.getMessage());
                this.downloadStats.addDeniedOntology(this.metaData.acronym);
                this.removeOntologyFiles();
            }
            catch (ResourceNotFoundException e) {
                log.warn("Resource not found for ontology {}: {}. Ontology is skipped.", (Object)this.metaData.acronym, (Object)e.getMessage());
                this.downloadStats.addOntologyWithDownloadError(this.metaData.acronym, e.getMessage());
                this.removeOntologyFiles();
            }
        }

        private void removeOntologyFiles() {
            log.info("Deleting info files for ontology {}", (Object)this.metaData.acronym);
            if (this.metaDataFile.exists()) {
                this.metaDataFile.delete();
            }
            if (this.submissionFile.exists()) {
                this.submissionFile.delete();
            }
            if (this.submissionsFile.exists()) {
                this.submissionsFile.delete();
            }
            if (this.projectsFile.exists()) {
                this.projectsFile.delete();
            }
            if (this.analyticsFile.exists()) {
                this.analyticsFile.delete();
            }
        }

        @Override
        public OntologyMetaData call() throws Exception {
            int maxRetries = 10;
            int retries = 0;
            while (retries < maxRetries) {
                try {
                    this.download();
                    return this.metaData;
                }
                catch (ResourceDownloadException | IOException e) {
                    try {
                        if (e.getMessage().contains("504")) {
                            log.info("{}: Error was a gateway timeout. Waiting an hour and then retry.", (Object)this.metaData.acronym);
                            Thread.sleep(3600000L);
                            ++retries;
                            continue;
                        }
                        log.info("{}: Server error occured: {}. Waiting an hour and then retry", (Object)this.metaData.acronym, (Object)e.getMessage());
                        Thread.sleep(3600000L);
                        ++retries;
                    }
                    catch (InterruptedException e1) {
                        e1.printStackTrace();
                    }
                }
                catch (JsonSyntaxException | ParseException e) {
                    e.printStackTrace();
                    break;
                }
            }
            if (retries == maxRetries) {
                log.error("Could not download complete data for ontology {} after {} retries. Aborting.", (Object)this.metaData.acronym, (Object)retries);
            }
            return null;
        }
    }
}

