package de.julielab.jssf.commons.spi;

import de.julielab.jssf.commons.util.ConfigurationException;
import org.apache.commons.configuration2.HierarchicalConfiguration;
import org.apache.commons.configuration2.XMLConfiguration;
import org.apache.commons.configuration2.builder.FileBasedConfigurationBuilder;
import org.apache.commons.configuration2.builder.fluent.Parameters;
import org.apache.commons.configuration2.io.FileBased;
import org.apache.commons.configuration2.io.FileHandler;
import org.apache.commons.configuration2.tree.ImmutableNode;
import org.apache.commons.configuration2.tree.xpath.XPathExpressionEngine;

import java.io.File;
import java.nio.charset.StandardCharsets;

public interface ConfigurationTemplateGenerator extends ParameterExposing{

    /**
     * This default implementation constructs an empty {@link XMLConfiguration} with an {@link XPathExpressionEngine}
     * and a UTF-8 encoding. Note that when using this template generation method, the keys of the configuration
     * must be given in XPath form, i.e. 'key/subkey' instead of the default dotted notation 'key.subkey'.
     * @return An empty XMLConfiguration template.
     * @throws ConfigurationException If the template generation fails.
     */
    default HierarchicalConfiguration<ImmutableNode> createConfigurationTemplate() throws ConfigurationException {
        Parameters params = new Parameters();
        FileBasedConfigurationBuilder<XMLConfiguration> builder =
                new FileBasedConfigurationBuilder<XMLConfiguration>(XMLConfiguration.class)
                        .configure(params.xml()
                                .setExpressionEngine(new XPathExpressionEngine())
                                .setEncoding(StandardCharsets.UTF_8.name())
                        );
        XMLConfiguration c;
        try {
            c = builder.getConfiguration();
            exposeParameters("", c);
        } catch (org.apache.commons.configuration2.ex.ConfigurationException e) {
            throw new ConfigurationException();
        }
        return c;
    }

    /**
     * Generates the configuration template by calling {@link #createConfigurationTemplate()} and stores the
     * template to <tt>destination</tt>.
     * @param destination The file path where the template should be written to.
     * @throws ConfigurationException It template generation of population fails.
     */
    default void writeConfigurationTemplate(File destination) throws ConfigurationException {
        try {
            HierarchicalConfiguration<ImmutableNode> template = createConfigurationTemplate();
            if (!(template instanceof FileBased))
                throw new ConfigurationException("The created configuration cannot be stored to file " +
                        "because the chosen configuration implementation " + template.getClass().getCanonicalName() + " " +
                        "does not implement the " + FileBased.class.getCanonicalName() + " interface");
            FileHandler fh = new FileHandler((FileBased) template);
            fh.save(destination);
        } catch (org.apache.commons.configuration2.ex.ConfigurationException e) {
            throw new ConfigurationException();
        }
    }
}
