package de.julielab.jssf.commons.services;

import de.julielab.jssf.commons.util.ResourceAccessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DownloadService implements IDownloadService {

    private final static Logger log = LoggerFactory.getLogger(DownloadService.class);

    private Matcher contentDispositionFilenameMatcher = Pattern.compile("filename=\"([^\"]+)\"").matcher("");

    @Override
    public File store(URI address, File destination, String downloadName) throws ResourceAccessException {
        String archiveName = downloadName == null ? "archive" : downloadName;
        try {
            log.info("Trying to retrieve {} from {}", archiveName, address);
            URL url = address.toURL();
            if (!destination.exists()) {
                log.debug("{} destination directory {} does not exist, necessary directories are created.", archiveName, destination);
                if (!destination.mkdirs())
                    throw new ResourceAccessException("Could not create directories " + destination.getAbsolutePath());
            }
            URLConnection connection = url.openConnection();
            InputStream is = connection.getInputStream();
            String fileName = getFilename(url, connection);
            log.debug("Storing the downloaded resource to {}", destination);
            File downloadedFile = new File(destination.getAbsolutePath() + File.separator + fileName);
            Files.copy(is, downloadedFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            return downloadedFile;
        } catch (ResourceAccessException e) {
            log.error("Error while trying to download the resource at " + address.toString());
            throw e;
        } catch (IOException e) {
            throw new ResourceAccessException(e);
        }
    }

    @Override
    public File store(URI uri, File destinationDirectory) throws ResourceAccessException {
       return store(uri, destinationDirectory, null);
    }


    /**
     * Synchronized because of the use of a matcher stored in a field. Default visibility for testing.
     *
     * @param url        The archive URL.
     * @param connection The URLConnection to retrieve the file name from.
     * @return The archive file name as extracted from the HTTP header or <tt>url.getFile()</tt>.
     */
    synchronized String getFilename(URL url, URLConnection connection) {
        String fileName = connection.getHeaderField("Content-Disposition");
        if (fileName != null) {
            contentDispositionFilenameMatcher.reset(fileName);
            if (contentDispositionFilenameMatcher.find())
                fileName = contentDispositionFilenameMatcher.group(1);
        }
        if (fileName == null) {
            fileName = new File(url.getFile()).getName();
        }
        log.debug("Extracted resource file name as: {}", fileName);
        return fileName;
    }
}
