#!/usr/bin/perl

use strict;
use warnings;

BEGIN 
{
    unshift @INC, './modules';
}

use Jls::File;
use Jls::TapCalculate;
use Jls::TapInput;
use Jls::TapMedianEvalue;
use Jls::TapOutput;

my @retrieval_lists = ();
my @ks = ();
my @thresholds = ();
my $threshold;
my $quantile = 0.5;

my $dir = '.';
my $unweighted = 0;
my $pad_0 = 0;
my $monotonicity = Jls::TapCalculate::equal ();
my $file = "";

my $str;
my $output = ''; # (STDOUT) file for output 

if ($#ARGV == -1) { error_args ('', -1); exit (0); }

for (my $arg = 0; $arg <= $#ARGV; $arg++)
{
    my $s = substr ($ARGV [$arg], 0, 2);

    if ($arg == $#ARGV) { error_args ($ARGV [$arg], 1); }
    
    if ($s eq '-u') { $unweighted = 1; } 
    elsif ($s eq '-p') { $pad_0 = 1; } 
    elsif ($s eq '-d')
    {
        $dir = $ARGV [++$arg];
    }
    elsif ($s eq '-a')
    {
        if ($output) { error_args ($ARGV [$arg], 2); }
        $output = $ARGV [++$arg];
        open(STDOUT, ">>$output") or die "Redirection of STDOUT to $output failed.";
    }
    elsif ($s eq '-f')
    {
        eval {
            Jls::TapInput::push_file_of_files_onto_retrieval_lists (\@retrieval_lists, $ARGV [++$arg]);
        };
        
        if ($@) { Jls::TapOutput::error_computation ($@, $ARGV [$arg]); }
    }
    elsif ($s eq '-i')
    {
		$file = $ARGV[++$arg];
        eval {
            Jls::TapInput::push_file_onto_retrieval_lists (\@retrieval_lists, $dir . '/' . $file);
        };
        
        if ($@) { Jls::TapOutput::error_computation ($@, $ARGV [$arg]); }
    }
    elsif ($s eq '-k')
    {
        while ($arg < $#ARGV) 
        {
            my $k = $ARGV [++$arg]; 
            push (@ks, $k); 
        }
    }
    elsif ($s eq '-m')
    {
        my $word = $ARGV [++$arg];
        
        if ($word =~ m/^asc/) { $monotonicity = Jls::TapCalculate::ascending (); }
        elsif ($word =~ m/^desc/) { $monotonicity = Jls::TapCalculate::descending (); }
        else { error_args ($ARGV [$arg], 2) }
    }
    elsif ($s eq '-q')
    {
        $quantile = $ARGV [++$arg];
    }
    elsif ($s eq '-t')
    {
        while ($arg < $#ARGV) 
        {
            my $score = $ARGV [++$arg]; 
            push (@thresholds, $score); 
        }
    }
    elsif ($s eq '-w')
    {
        if ($output) { error_args ($ARGV [$arg], 2); }
        $output = $ARGV [++$arg];
        open(STDOUT, ">$output") or die "Redirection of STDOUT to $output failed.";
    }
    else { error_args ($ARGV [$arg], 0); }
}

if ($#retrieval_lists == -1) { error_args ('[There are no retrieval lists.]', 2); }

eval { Jls::TapCalculate::precompute_for_taps (\@retrieval_lists, \$monotonicity); };
if ($@) { Jls::TapOutput::error_computation ($@); }

if ($#thresholds == -1)
{
    if ($#ks == -1) { error_args ('', 3); }
    
    if (! Jls::File::is_float ($quantile) || $quantile <= 0.0 || 1.0 < $quantile) 
    { 
        error_args ($quantile, 2); 
    }

    foreach my $k (@ks)
    {
        if (! Jls::File::is_integer ($k) || $k <= 0) { error_args ($k, 2); }
        
        eval {
            push (@thresholds, 
                  Jls::TapMedianEvalue::median_threshold_for_retrieval_lists 
                  (\@retrieval_lists, $k, $quantile, $unweighted, $pad_0));
        };

        if ($@) { Jls::TapOutput::error_computation ($@, $k); }
    }
}
else 
{
    foreach $threshold (@thresholds)
    {
        if (! Jls::File::is_float_or_NaN ($threshold)) { error_args ($threshold, 2); }
        elsif ($threshold eq 'NaN' && ! $pad_0) { error_args ($threshold, 2); }
    }
}

# Computes the taps.

my @taps = ();

foreach my $threshold (@thresholds)
{
    eval 
    { 
        push (@taps, Jls::TapCalculate::taps_for_retrieval_lists (\@retrieval_lists, $threshold, $unweighted));
    };
    
    if ($@) { Jls::TapOutput::error_computation ($@, $threshold); }
};

$file =~ s/(.*)\.txt\.tap\.in/$1/;
print Jls::TapOutput::tap_out (\@taps, \@ks, $quantile, $unweighted, $file);

sub args
{
my $msg = '
Program options are arguments with \'-\' followed by a letter.
An option requiring further input(s) appears with square brackets.
An option with no input always has a default; it lacks square brackets.
The default for an option, if any, is indicated in parentheses.

-a [output file, for appending] (default : STDOUT)
-d [the directory path to the input file(s) in option -i] (default : '.')]
-f [a single input file containing the input file(s) with retrieval list(s)]
-i [a single input file containing retrieval list(s)]
-k [multiple k\'s, the final arguments, giving the quantile q errors per query]
-m [monotonicity of scores: "asc" or "desc"] (default : the lists determine)
-p pad insufficient retrieval list(s) with irrelevant records
-q [quantile q, 0.0 < q <= 1.0] (default : the median, 0.5)
-t [multiple threshold scores or E-values, the final arguments]
-u ignore all weights, to perform an unweighted calculation
-w [output file, for overwriting] (default : STDOUT)';
    
    return $msg;
}

# Reports an error in an argument.

sub error_args
# $arg_
# $code_ 
# // 0 = no known option
# // 1 = final option was bare
# // 2 = the argument does not satisfy required constraints
# // 3 = threshold and k both missing
{
    (my $arg_, my $code_) = @_;
 
    my $msg = "\n\n" . Jls::TapOutput::program_name ();
    
    if ($code_ == 0) 
    {
        $msg .= "\n\n" . 'The following argument is an unknown option.';
        $msg .= "\n" . $arg_;
    }
    elsif ($code_ == 1) 
    {
        $msg .= "\n\n" . 'The following argument parses as an option';
        $msg .= ' without any input following it.';
        $msg .= "\n" . $arg_;
     }
    elsif ($code_ == 2) 
    {
        $msg .= "\n\n" . 'The following argument does not satisfy required constraints.';
        $msg .= "\n" . $arg_;
    }
    
    elsif ($code_ == 3) 
    {
        $msg .= "\n\n" . 'One of the options -t or -k must be present.';
    }
    
    $msg .= "\n" . args ();
    $msg .= "\n" . Jls::TapOutput::url ();

    if ($code_ == -1) { print $msg . "\n"; }
    else { die $msg . "\n\nDied"; }
}
