#!/usr/bin/perl

use strict;
use warnings;

package Jls::TapOutput;

use Exporter;
our @ISA = 'Exporter';
our @EXPORT = 'error_computation url program_name tap_out';

=head 1

# Reports an error in an computation.

sub error_computation
# $msg_
# $arg_

# Directs the user to the help URL.

sub url

# Returns a string of ascii output of TAPs.

sub tap_out
# \@taps_ // an array reference to the TAP's
# \@ks_ // an array reference to the corresponding k's
# $quantile_ // (0.5)

=cut

# Reports an error in an computation.

sub error_computation
# $msg_
# $arg_
{
    (my $msg_, my $arg_) = @_;
    
    my $msg = "\n\n" . program_name ();
    $msg .= "\n";
        
    $msg .= "\n" . 'The TAP computation threw an error.';
    $msg .= "\n" . $msg_;
    
    if ($arg_) 
    { 
        $msg .= "\n" . 'The following argument or quantity contributed to the error.';
        $msg .= "\n" . $arg_;
    }
    
    $msg .= "\n" . url ();

    die $msg . "\n\nDied";
}

# Directs the user to the help URL.

sub url
{
my $msg = '
Please see the URL
http://www.ncbi.nlm.nih.gov/CBBresearch/Spouge/html.ncbi/tap/help.html
for further information.';
    
    return $msg;
}

sub program_name
{
    return 'tap.pl Version 1.3';
}

# Returns a string of ascii output of TAPs.

sub tap_out
# \@taps_ // an array reference to the TAP's
# \@ks_ // an array reference to the corresponding k's
# $quantile_ // (0.5)
# $unweighted_ : (0) ? Enforce an unweighted calculation ?
{
    (my $taps_, my $ks_, my $quantile_, my $unweighted_, my $file) = @_;

    if (! $quantile_) { $quantile_ = 0.5; }
    if (! $unweighted_) { $unweighted_ = 0; }
    
    my $k_index = 0;
    my $str;
    my $output = "\n\n" . program_name ();
            
    foreach my $tap (@$taps_)
    {
        $output .= "\n\n";
		$output .= "GN Input file\t";
        
        if (scalar (@$ks_) != 0) { $output .= "EPQ (Threshold for $quantile_ Quantile)"; }
        else { $output .= "Threshold"; }
    
        $output .= "\t";
        
        if ($unweighted_ == 0 && $tap->{'is_weighted'} == 1) 
        { 
            $output .= "Weighted "; 
        } 
        else { $output .= "Unweighted "; } 
            
        $output .= 'Average TAP';
        $output .= "\n";
		$output .= $file."\t";
    
        if (scalar (@$ks_) != 0) 
        { 
            $output .= $ks_->[$k_index++] . ' (' . $tap->{'threshold'} . ')'; 
        }
        else { $output .= $tap->{'threshold'}; }
    
        $str = sprintf ("%.4f", $tap->{'average_tap'});
        $output .= "\t" . $str;
    
        if ($#{$tap->{taps}} != -1)
        {
            $output .= "\n";
            $output .= "\nIndividual Retrieval Lists";
            
            if ($unweighted_ == 0 && $tap->{'is_weighted'} == 1) 
            { 
                $output .= " (with Weights)"; 
            }
             
            $output .= " and Their TAPs";
            
            for (my $i = 0; $i <= $#{$tap->{taps}}; $i++)
            {
                $str = sprintf ("%.4f", $tap->{taps} [$i]->{'tap'});
        
                $output .= "\n" . $tap->{taps} [$i]->{'id'};
                if ($unweighted_ == 0 && $tap->{'is_weighted'} == 1) 
                { 
                    $output .= " (" . $tap->{taps} [$i]->{'weight'} . ')'; 
                }
                $output .= "\t" . $str;
            }
        }
    }

$output .= "

In any work or product based on this material, please cite 

H.D. Carroll, M.G. Kann, S.L. Sheetlin, and J.L. Spouge
\"Threshold Average Precision (TAP-k): A Measure of Retrieval Efficacy Designed for Bioinformatics\"
(2010) Bioinformatics 26 : 1708-1713
";

    return $output;
}

1;
