#!/usr/bin/perl

use strict;
use warnings;

package Jls::PoissonInfection;

use Exporter;
our @ISA = 'Exporter';
our @EXPORT = 'new push push_vector complete get_pvalue get_likelikehood_ratio_pvalue';

=head1 NAME

Jls::PoissonInfection - Perl module for calculating the statistics and p-values under a Poisson null model for infection.

=head1 SYNOPSIS

use Jls::PoissonInfection;

$p=PoissonInfection->new (); # Creates a new object.

my %h_ = 
( 
    {'d'} => 10.0  # viral dosage (concentration)
    {'n'} => 5,    # number of animals
    {'i'} => 2,    # number of animals infected
);

$p->push (\%h_); # Pushes a hash reference \%h onto the data.

@d = (1, 1, 2, 5, 11); # viral dosage (concentration)

@n = (6, 1, 2, 1, 1);  # number of animals

@i = (6, 0, 0, 0, 0);  # number of animals infected

$p->push_vector (\@d, \@n, \@i); # Pushes vectors onto the data.

$p->complete ();

# Completes the object by calculating the parameters of infection under a Poisson model.
# Completion is required before calling any of the following routines.

$p->get_mean_log10_iu (); # Returns the mean of log10 infectious units.

$p->get_stdev_log10_iu () # Returns the standard deviation of log10 infectious units.

$p->get_pvalue (); # Returns the pvalue of a Poisson model.

$p->get_likelikehood_ratio_pvalue ($p0);

# Returns the likelikehood ratio pvalue for adding to a completed PoissonInfection object.

=head1 DESCRIPTION

For more information, see: L<http://www.ncbi.nlm.nih.gov/CBBresearch/Spouge/html.ncbi/index/software.html#1>.

=head1 BUGS 

This version has been released after testing against examples in Siegel and Castellan. Nevertheless, if you find any bugs or oddities, please do inform the author. 

=head1 INSTALLATION 

See perlmodinstall for information and options on installing Perl modules. 

=head1 AVAILABILITY 

The latest version of this module is available from John L. Spouge.

=head1 AUTHOR

John L. Spouge <spouge@ncbi.nlm.nih.gov>

=head1 COPYRIGHT 

Copyright 2010 John L. Spouge. All rights reserved. 

This library is free software; you can redistribute it and/or modify it under the same terms as Perl itself. 

=head1 SEE ALSO

Statistics::Distribution (Copyright 2003 Michael Kospach. All rights reserved.), perl(1).
Math::Function::Roots (Copyright 2005 Spencer Ogden, All Rights Reserved.), perl(1).

=cut

# Creates a new object.

sub new
{
    my $class_ = shift;
    
    my @a = (); 

    my $self = 
    {
        array => \@a,
    };
    
    return bless $self, $class_;
}

# Returns the mean of log10 infectious units.

sub get_mean_log10_iu
{
    return $_ [0]->{'mean_log10_iu'};
}

# Returns the standard deviation of log10 infectious units.

sub get_stdev_log10_iu
{
    return $_ [0]->{'stdev_log10_iu'};
}

# Returns the pvalue of a Poisson model.

sub get_pvalue
{
    return $_ [0]->{'pvalue'};
}

# Pushes a hash reference \%h onto the data.

sub push
# \%h_ // a reference to a hash triple
{
    my $class_ = shift;
    my $h_ = shift;
    
    my $a_ = $class_ ->_get_array ();

    my %h = 
    (
        d => $h_->{'d'},
        n => $h_->{'n'},
        i => $h_->{'i'},
    );
     
    push (@$a_, \%h);
    
    return $class_;
}

# Pushes a hash reference \%h onto the data.

sub push_vector
# \@d_ // an array reference to dosages
# \@n_ // an array reference to numbers of animals
# \@i_ // an array reference to numbers of infected animals
{
    my $class_ = shift;
    my $d_ = shift;
    my $n_ = shift;
    my $i_ = shift;
    
    (scalar (@$d_) == scalar (@$n_)) || 
        die 'PoissonInfection::push_vector : ! (scalar (@$d_) == scalar (@$n_))';
    (scalar (@$n_) == scalar (@$i_)) || 
        die 'PoissonInfection::push_vector : ! (scalar (@$n_) == scalar (@$i_))';
    
    my %h = ();

    for (my $j = 0; $j < scalar (@$d_); $j++)
    {
        $h {'d'} = $d_->[$j];
        $h {'n'} = $n_->[$j];
        $h {'i'} = $i_->[$j];
    
        $class_->push (\%h);
    }
    
    return $class_;
}

# Returns the likelikehood ratio pvalue for adding to a completed PoissonInfection object.

sub get_likelikehood_ratio_pvalue
# $p_ // another completed PoissonInfection object
{
    my $class_ = shift;
    my $p_ = shift;
    
    my $h0 = new Jls::PoissonInfection ()->_add ($class_)->_add ($p_)->complete ();

    my $d = $class_->_get_lambda ();
    $d += $p_->_get_lambda ();
    $d -= $h0->_get_lambda ();
    $d *= 2.0;

    use Statistics::Distributions;

    return Statistics::Distributions::chisqrprob (1.0, $d);
}

# Calculates the parameters of infection under a Poisson model.

sub complete
{
    my $class_ = shift;
    my $ca = $class_->_get_array ();
    my @h = @$ca;
    
    @h = sort { $a->{'d'} <=> $b->{'d'} } @h;

    my $is_equal = 0;
    my $start = 0;
    
    for (my $j = 1; $j < scalar (@h); $j++)
    {        
        if ($h [$j]->{'d'} == $h [$j - 1]->{'d'}) 
        {
            if (! $is_equal) { $start = $j - 1; }
            $is_equal = 1;
            
            $h [$start]->{'n'} += $h [$j]->{'n'};
            $h [$start]->{'i'} += $h [$j]->{'i'};
            
            $h [$j]->{'n'} = 0;
            $h [$j]->{'i'} = 0;
        }
        else { $is_equal = 0; }
    }
    
    @$ca = ();
    
    for (my $j = 0; $j < scalar (@h); $j++)
    {        
        if ($h [$j]->{'n'} != 0) { $class_->push ($h [$j]); }
    }

    my $a_hat = $class_->_get_mle ();
    my $fisher_information = $class_->_calc_lambda_derivative_two_a_hat ($a_hat);
    
    my $lower = $class_->_get_lower ();
    my $upper = $class_->_get_upper ();
    my $lambda = $class_->_calc_lambda ($a_hat);
    
    my $deviance = $class_->_get_saturated_log_likelihood ($lower, $upper);
    $deviance -= $lambda;
    $deviance *= 2.0;

    use Statistics::Distributions;

    $class_->{'mean_log10_iu'} = -log ($a_hat) / log (10.0);
    $class_->{'stdev_log10_iu'} = 1.0 / sqrt ($fisher_information) / log (10.0);
    $class_->{'pvalue'} = Statistics::Distributions::chisqrprob ($upper - $lower, $deviance);
    $class_->{'lambda'} = $lambda;
    
    return $class_;
}

sub _add
# $p_ // another PoissonInfection object
{
    my $class_ = shift;
    my $p_ = shift;
    
    my $a_ = $p_->_get_array ();
    
    for (my $j = 0; $j < scalar (@$a_); $j++)
    {
        $class_->push ($a_->[$j]);
    }
    
    return $class_;
}

# Returns the log-likelihood of a Poisson model.

sub _get_lambda
{
    return $_ [0]->{'lambda'};
}

sub _get_array
{
    return $_ [0]->{'array'}; 
}

# Returns the smallest index without infection.

sub _get_lower
{
    my $h_ = $_ [0]->_get_array ();

    my $j_caps = scalar (@$h_);
    my $lower = 0;

    for (my $j = 0; $j < $j_caps; $j++)
    {
        if ($h_->[$j]->{'i'} != 0) { last; }; 
        $lower = $j;
    }
    
    return $lower;
}

# Returns the largest index with complete infection.

sub _get_upper
{
    my $h_ = $_ [0]->_get_array ();

    my $j_caps = scalar (@$h_);
    my $upper = $j_caps;    

    for (my $j = 0; $j < $j_caps; $j++)
    {
        if ($h_->[$j_caps - 1 - $j]->{'i'} != $h_->[$j_caps - 1 - $j]->{'n'}) { last; }; 
        $upper = $j_caps - $j;
    }
    
    return $upper;
}

sub _calc_lambda
# $a_ // argument
# $lower_ // (0) sum is over [$lower_, $upper_)
# $upper_ // (scalar ($_ [0]->_get_array ()))
{
    my $class_ = shift;
    my $h_ = $class_->_get_array ();

    my $a_ = shift;
    ($a_ > 0.0) || die 'PoissonInfection::_calc_lambda : ! ($a_ > 0.0)';

    my $lower = $#_ == -1 ? 0 : shift;
    my $upper = $#_ == -1 ? scalar (@$h_) : shift;    

    my $s = 0.0;

    for (my $j = $lower; $j < $upper; $j++)
    {
        $s += $h_->[$j]->{'i'} * log (1.0 - exp (-$a_ * $h_->[$j]->{'d'})); 
        $s -= $a_ * $h_->[$j]->{'d'} * ($h_->[$j]->{'n'} - $h_->[$j]->{'i'}); 
    }
    
    return $s;
}

# Returns the derivative wrt log ($a_) under a Poisson model.

my $h_s = 0;

sub _lambda_derivative 
# $a_ // argument
{
    use Math::BigFloat;
    
    my $a_ = $_ [0];
    
    if ($a_ == 0.0) { return Math::BigFloat->binf (); }
    ($a_ > 0.0) || die 'PoissonInfection::_lambda_derivative : ! ($a_ > 0.0)';
    
    my $s = 0.0;
    
    for (my $j = 0; $j < scalar (@$h_s); $j++)
    {
        ($h_s->[$j]->{'d'} != 0.0) || 
        die 'PoissonInfection::_lambda_derivative : ! ($h_s->[$j]->{\'d\'} != 0.0)';
        
        $s += $h_s->[$j]->{'d'} * 
              ($h_s->[$j]->{'i'} / (exp ($a_ * $h_s->[$j]->{'d'}) - 1.0) - 
              ($h_s->[$j]->{'n'} - $h_s->[$j]->{'i'}));
    }
    
    $s *= $a_; # derivative wrt log ($a_)
    
    return $s;
}

sub _get_mle
{
    $h_s = $_ [0]->_get_array ();
    
    my $num = 0.0;
    my $den = 0.0;

    for (my $j = 0; $j < scalar (@$h_s); $j++)
    {
        $num += $h_s->[$j]->{'i'}; 
        $den += $h_s->[$j]->{'d'} * ($h_s->[$j]->{'n'} - $h_s->[$j]->{'i'}); 
    }
    
    ($num != 0.0) || die 'PoissonInfection::_get_mle : ! ($num != 0.0)';
    ($den != 0.0) || die 'PoissonInfection::_get_mle : ! ($den != 0.0)';
    
    use Math::Function::Roots;
    
    return Math::Function::Roots::bisection (\&_lambda_derivative, 0.0, $num / $den);
}

# Returns the derivative wrt log ($a_) under a Poisson model.

sub _calc_lambda_derivative
# $a_ // argument
{
    $h_s = $_ [0]->_get_array ();
    
    return _lambda_derivative ($_ [1]); 
}

# Returns the second derivative wrt log ($a_) at $a_hat under a Poisson model.

sub _calc_lambda_derivative_two_a_hat
# $a_ // argument $a_hat
{
    my $h_ = $_ [0]->_get_array ();
    my $a_ = $_ [1];
    ($a_ > 0.0) || die 'PoissonInfection::_calc_lambda_derivative_two_a_hat : ! ($a_ > 0.0)';

    my $s = 0.0;
    my $e = 0.0;

    for (my $j = 0; $j < scalar (@$h_); $j++)
    {
        $e = exp ($a_ * $h_->[$j]->{'d'}) - 1.0;

        $s += $h_->[$j]->{'d'} * $h_->[$j]->{'d'} * 
              $h_->[$j]->{'i'} * exp ($a_ * $h_->[$j]->{'d'}) / $e / $e;
    }
    
    $s *= $a_ * $a_;
    
    return $s;
}

sub _get_saturated_log_likelihood
# $lower_ // (0) sum is over [$lower_, $upper_)
# $upper_ // (scalar ($_ [0]->_get_array ()))
{
    my $class_ = shift;
    my $h_ = $class_->_get_array ();

    my $lower = $#_ == -1 ? 0 : shift;
    my $upper = $#_ == -1 ? scalar (@$h_) : shift;    

    my $s = 0.0;
    my $p = 0.0;
    my $q = 0.0;

    for (my $j = $lower; $j < $upper; $j++)
    {
        $p = $h_->[$j]->{'i'} / $h_->[$j]->{'n'};
        $q = 1.0 - $p;

        if ($p != 0.0 && $q != 0.0)
        {
            $s += $h_->[$j]->{'n'} * ($p * log ($p) + $q * log ($q));
        }
    }
    
    return $s;
}

1;
