#!/usr/bin/perl

use strict;
use warnings;

package Jls::File;

use Exporter;
our @ISA = qw/Exporter/;
our @EXPORT = 'file_to_string file_handle_to_string is_01 is_blank is_boolean is_integer is_float is_float_or_NaN dos';

=head 1

# Converts an input file to a string.

sub file_to_string 
# $input_ // Input file

# Converts an file-handle from a file upload box to a string.

sub file_handle_to_string 
# $file_ // Input file-handle

# Is the string blank?

sub is_blank

# ? Is the $string_ an integer?

sub is_01
# $string_

# ? Is the $string_ a boolean?

sub is_boolean
# $string_

sub is_integer
# $string_

# ? Is the $string_ a real number ?

sub is_float
# $string_

# ? Is the $string_ a real number or a 'NaN'?

sub is_float_or_NaN
# $string_

# Takes care of DOS returns & backslashes in a string with a directory.

sub dos
# $string_
# $quote_

=cut

# Converts an input file to a string.

sub file_to_string 
# $file_ // Input file
{
    my $file_ = $_ [0];
    
    my $msg = "Input file '" . $file_ . "' failed to open.\n" . "Died";

    my $terminator = $/;
    undef $/;
    open INPUT, "<$file_" or die $msg;
    my $str = <INPUT>; # terminator undefined : $str is the whole file.
    close INPUT;

    $/ = $terminator; # Restore for normal behavior later

    return $str;
}

# Converts an file-handle from a file upload box to a string.

sub file_handle_to_string 
# $file_ // Input file-handle
{
    my $file_ = $_ [0];

    my $terminator = $/;
    undef $/;
    my $str = <$file_>; # terminator undefined : $str is the whole file.

    $/ = $terminator; # Restore for normal behavior later

    return $str;
}

# Takes care of DOS returns & backslashes in a string with a directory.

sub dos
# $string_
# $quote_
{
    (my $string_, my $quote_) = @_;

    my $msg = $string_;
    $msg =~ s|\\|/|gs; # Takes care of DOS backslashes.
    $msg =~ s|\r{0,1}\n|\\n|gs; # Takes care of DOS newlines.
    
    if ($quote_)
    {
        $msg =~ s|$quote_|\\$quote_|gs; # Takes care of quotation marks.
    }
    
    return $msg;
}

# Is the string blank?

sub is_blank
# $string_
{
    my $string_ = $_ [0];
    if ($string_ =~ m/^\s*$/) { return 1; }

    return 0;
}

# ? Is the $string_ 0 or 1 ?

sub is_01
# $string_
{
    my $string_ = $_ [0];

    if ($string_ =~ m/^0$/) { return 1; }
    if ($string_ =~ m/^1$/) { return 1; }

    return 0;
}

# ? Is the $string_ a boolean ?

sub is_boolean
# $string_
{
    my $string_ = $_ [0];

    if ($string_ =~ m/^false$/i) { return 1; }
    if ($string_ =~ m/^f$/i) { return 1; }
    if ($string_ =~ m/^0$/) { return 1; }

    if ($string_ =~ m/^true$/i) { return 1; }
    if ($string_ =~ m/^t$/i) { return 1; }
    if ($string_ =~ m/^1$/) { return 1; }

    return 0;
}

# ? Is the $string_ an integer ?

sub is_integer
# $string_
{
    my $string_ = $_ [0];

    return $string_ =~ m/^(\+|\-)?\d*$/ ;
}

# ? Is the $string_ a real number ?

sub is_float
# $string_
{
    my $string_ = $_ [0];

    if ($string_  =~ m/^\D+$/) { return ''; }

    return $string_ =~ m/^(\+|\-)?\d*\.?\d*(e(\+|\-)?\d+)?$/i ;
}

# ? Is the $string_ a real number or a 'NaN'?

sub is_float_or_NaN
# $string_
{
    my $string_ = $_ [0];

    if (is_float ($string_)) { return is_float ($string_); }

    return $string_ eq 'NaN' ? 'NaN' : '';
}

1;
